#extension GL_ARB_gpu_shader_int64 : enable
#extension AMD_gpu_shader_int64 : enable

//Fragment Output
layout(location = 0) out uvec3 outColor;
layout(location = 1) out vec3 outLuminance;

//Samplers
uniform usampler2D colortex0;
uniform sampler2D colortex1;
uniform usampler2D colortex2;
uniform usampler2D colortex3;
uniform sampler2D colortex5;
uniform sampler2D colortex6;
uniform sampler2D depthtex0;

uniform sampler2D gaux4;

uniform sampler2D noisetex;

uniform sampler2D shadowtex0;
uniform sampler2D shadowtex1;
uniform sampler2D shadowcolor0;
uniform sampler2D shadowcolor1;

//Sky Textures
uniform sampler2D depthtex1;
uniform sampler2D depthtex2;

//Uniforms
uniform mat4 gbufferProjection, gbufferModelView;
uniform mat4 gbufferProjectionInverse, gbufferModelViewInverse;

uniform mat4 shadowProjection, shadowModelView;
uniform mat4 shadowProjectionInverse, shadowModelViewInverse;

uniform vec3 sunVector, moonVector, shadowLightVector, upVector;
uniform vec3 sunVectorView, moonVectorView, shadowLightVectorView;
uniform vec3 cameraPosition;

uniform vec2 viewSize, viewPixelSize;

uniform ivec2 eyeBrightness;

uniform float eyeAltitude, frameTimeCounter;

uniform int frameCounter, isEyeInWater;

//Fragment Inputs
struct lightStruct {
    vec3 indirect;
    vec3 direct;

    vec3 skySh[9];
};

in lightStruct lc;
in vec2 textureCoordinate;

#include "/lib/universal/universal.glsl"
#include "/lib/fragment/dither/bayer.glsl"
#include "/lib/fragment/dither/hash.glsl"
#include "/lib/fragment/dither/hammersley.glsl"
#include "/lib/fragment/dither/randNext.glsl"
#include "/lib/fragment/colorspace.glsl"

#include "/lib/fragment/positionDepth.glsl"
#include "/lib/fragment/surfaceData.glsl"

#include "/lib/fragment/lighting/direct/distortion.glsl"
#include "/lib/fragment/lighting/direct/hard.glsl"
#include "/lib/fragment/lighting/direct/soft.glsl"

#include "/lib/ray/rayShape.glsl"
#include "/lib/ray/raytracer.glsl"

#include "/lib/shared/sky/celestial/constants.glsl"

#include "/lib/shared/sky/constants.glsl"
#include "/lib/shared/sky/phase.glsl"
#include "/lib/shared/sky/projection.glsl"
#include "/lib/shared/sky/atmosphere/lookup.glsl"
#include "/lib/shared/sky/atmosphere/transmittance.glsl"

#include "/lib/shared/sky/clouds/volumetric/constants.glsl"
#include "/lib/shared/sky/clouds/volumetric/functions.glsl"
#include "/lib/shared/sky/clouds/volumetric/shadow.glsl"

#include "/lib/shared/surface/water/constants.glsl"
#include "/lib/shared/surface/water/waves.glsl"

#include "/lib/fragment/lighting/lighting.glsl"
#include "/lib/fragment/lighting/volumetric.glsl"

#include "/lib/fragment/lighting/brdf/specular/ggx.glsl"

#include "/lib/shared/sky/celestial/celestialObjects.glsl"

vec3 hitColorNoInterpolation(in vec2 coordinate) {
    vec4 encodedColor = vec4(unpackUnorm2x16(texelFetch(colortex0, ivec2(coordinate * viewSize), 0).r), unpackUnorm2x16(texelFetch(colortex0, ivec2(coordinate * viewSize), 0).g));
    return decodeRGBE8(encodedColor);
}

vec3 calculateWaterReflections(in vec3 background) {
    vec3 direction = reflect(normalize(pd.viewPosition[0]), sd.viewNormals);
    float cosTheta = dot(direction, sd.viewNormals);
    complexVec3 n1 = complexVec3(vec3(1.00029), vec3(0.0));
    complexVec3 n2 = complexVec3(vec3(1.3310, 1.3330, 1.3374), vec3(2.4540e-8, 1.9600e-9, 1.1320e-9));
    if(isEyeInWater == 1) {
        n1 = complexVec3(vec3(1.3310, 1.3330, 1.3374), vec3(2.4540e-8, 1.9600e-9, 1.1320e-9));
        n2 = complexVec3(vec3(1.00029), vec3(0.0));
    }
    vec3 fresnel = fresnelNonPolarized(cosTheta, n1, n2);

    vec3 hitPosition;
    bool hit = raytraceIntersection(depthtex1, pd.screenPosition[0], direction, hitPosition, 64u, 16.0, false);

    vec3 reflection = vec3(0.0);
    if(hit) {
        reflection = hitColorNoInterpolation(hitPosition.xy).rgb;
    } else {
        reflection = texture(colortex5, projectSky(mat3(gbufferModelViewInverse) * direction)).rgb * sd.lightmap.y;
    }

    if(isEyeInWater == 1) {
        reflection = calculateWaterFog(vec3(1.0), direction * 1000.0, vec3(0.0));
    }

    vec3 celestialReflection = vec3(0.0);
    if(isEyeInWater == 0 && !hit) {
        vec3 reflectionDirection = mat3(gbufferModelViewInverse) * direction;
        vec3 transmittanceView = atmosphereTransmittance(vec3(0.0, planetRadius, 0.0), reflectionDirection);
        celestialReflection = (physicalSun(reflectionDirection) + physicalMoon(reflectionDirection)) * transmittanceView;
        celestialReflection *= calculateHardShadows(pd.shadowClipPosition);
        celestialReflection *= getCloudShadows(pd.scenePosition[0]);
    }

    return mix(background, reflection, fresnel) + celestialReflection;
}

vec3 calculateWaterRefraction() {
    float n1 = 1.00029;
    float n2 = 1.33333;

    #ifdef RAYTRACE_REFRACTION
        if(isEyeInWater == 1) {
            n1 = 1.33333;
            n2 = 1.00029;
        }

        vec3 direction = refract(fNormalize(pd.viewPosition[0]), sd.viewNormals, n1/n2);
    #else
        vec3 direction = refract(fNormalize(pd.viewPosition[0]), sd.viewFlatNormals - sd.viewNormals, n1/n2);
    #endif

    #ifndef RAYTRACE_REFRACTION
        float refractionAmount = distance(pd.viewPosition[1], pd.viewPosition[0]);
            refractionAmount = saturate(refractionAmount);

        vec3 hitPosition = direction * refractionAmount + pd.viewPosition[0];

        vec3 hitCoordinate = viewSpaceToScreenSpace(hitPosition, gbufferProjection);

        hitCoordinate.z = texture(depthtex1, hitCoordinate.xy).r;

        if(hitCoordinate.z < pd.depth0 || floor(hitCoordinate.xy) != vec2(0.0)) {
            hitCoordinate = pd.screenPosition[1];
        }
    #else
        vec3 hitPosition;
        bool hit = raytraceIntersection(depthtex1, pd.screenPosition[0], direction, hitPosition, RAYTRACE_REFRACTION_STRIDE, 16.0, false);

        vec3 hitCoordinate = hitPosition;
        hitCoordinate.z = texture(depthtex1, hitPosition.xy).r;

        if(!hit && isEyeInWater == 0) {
            hitCoordinate = pd.screenPosition[1];
        }
    #endif

    vec3 endRefractPosition = hitCoordinate;
         endRefractPosition = screenSpaceToViewSpace(endRefractPosition, gbufferProjectionInverse);

    return isEyeInWater > 0 ? hitColorNoInterpolation(hitCoordinate.xy) : calculateVolumetricWaterFog(hitColorNoInterpolation(hitCoordinate.xy).rgb, endRefractPosition, pd.viewPosition[0]);
}

vec3 calculateSurfaceReflections(in vec3 background) {
    if(sd.f0 > 0.0) {
        background *= (1.0 - sd.metalness);
        vec3 viewDirection = fNormalize(pd.viewPosition[0]);
        vec3 reflection = vec3(0.0);
        vec3 torchColor = blackbody(2000.0) * 8e2;
        uint n = 1;
        for(uint i = 0; i < n; ++i) {
            vec3 facetNormal = generateFacetNormal(sd.viewNormals, pd.viewPosition[0], sd.roughness);
            vec3 direction = reflect(viewDirection, facetNormal);
            float cosTheta = dot(-viewDirection, facetNormal);
            vec3 fresnel = fresnelFunction(cosTheta, sd.n, sd.k);

            vec3 hitPosition;
            vec3 startPosition = pd.screenPosition[0];
            bool hit = raytraceIntersection(depthtex1, startPosition, direction, hitPosition, uint(mix(32.0, 512.0, sd.roughness)), 16.0, false);

            if(hit) {
                uvec4 data1 = texture(colortex3, hitPosition.xy);
                vec4 specularData = vec4(unpackUnorm2x16(data1.r), unpackUnorm2x16(data1.g));

                if(specularData.g > 0.9) {
                    uvec4 data0 = texture(colortex2, hitPosition.xy);
                    vec3 viewNormals = mat3(gbufferModelView) * DecodeUnitVector(unpackSnorm2x16(data0.g));
                    vec3 viewDirection = fNormalize(screenSpaceToViewSpace(hitPosition, gbufferProjectionInverse));
                    vec3 direction = reflect(viewDirection, viewNormals);
                    reflection += (texture(colortex5, projectSky(mat3(gbufferModelViewInverse) * direction)).rgb * sd.lightmap.y) * fresnel;
                } else {
                    reflection += hitColorNoInterpolation(hitPosition.xy).rgb * fresnel;
                }
            } else {
                reflection += (texture(colortex5, projectSky(mat3(gbufferModelViewInverse) * direction)).rgb * sd.lightmap.y) * fresnel;
                if(sd.metalness > 0.9) reflection += (torchColor * (sd.lightmap.x * (1.0 - sd.lightmap.y))) * fresnel;
            }

            float reflectionContribution = smithGGXMaskingShadowing(dot(sd.viewNormals, -viewDirection), dot(sd.viewNormals, direction), sd.roughnessSquared) / smithGGXMasking(dot(sd.viewNormals, -viewDirection), sd.roughnessSquared);
            reflectionContribution = saturate(reflectionContribution);

            reflection *= reflectionContribution;
        } reflection = max(reflection, 0.0);
        reflection /= n;

        float nDotV = dot(sd.viewNormals, -viewDirection);
        float nDotL = saturate(dot(sd.viewNormals, shadowLightVectorView));
        float nDotH = saturate(dot(sd.viewNormals, fNormalize(-viewDirection + shadowLightVectorView)));
        float vDotH = dot(-viewDirection, fNormalize(-viewDirection + shadowLightVectorView));

        vec3 lightReflection = specularBRDF(nDotL, nDotV, nDotH, vDotH) * lc.direct;
        if(dot(sd.viewFlatNormals, shadowLightVectorView) < 0.0) {
            lightReflection = vec3(0.0);
        } else {
            lightReflection *= calculateSoftShadows(pd.shadowClipPosition, false) / pi;
            lightReflection *= getCloudShadows(pd.scenePosition[0]);
            float waterDepth;
            float waterFraction;
            bool castWater = waterShadow(pd.shadowClipPosition, waterDepth, waterFraction);
            if(castWater && waterDepth > 0.2) {
                vec3 waterTransmittance = vec3(0.0);
                waterDepth = waterDepth - 0.2;
                for(int n = 0; n < 3; ++n) {
                    waterTransmittance += exp(-(waterAttenuationCoefficient * waterDepth) * pow(0.50, n)) * pow(0.50, n);
                }
                lightReflection *= waterTransmittance/2.0;
            }
        }

        return background + reflection + lightReflection;
    } else {
        return background;
    }
}

/* DRAWBUFFERS:05 */
void main() {
    waterCoefficients();

    vec2 coord = mod(gl_FragCoord.xy, 512);
	uint s = uint(coord.x * viewSize.y + coord.y);
	     s = s * 720720u + uint(frameCounter);
	init_msws(s);

    fillSurfaceStruct();
    fillPositionStruct();
    vec4 encodedColor = vec4(unpackUnorm2x16(texture(colortex0, textureCoordinate).r), unpackUnorm2x16(texture(colortex0, textureCoordinate).g));
    vec3 texColor = decodeRGBE8(encodedColor);

    if(landmask(pd.depth0)) {
        vec4 texTransparent = texture(colortex1, textureCoordinate);
             texTransparent.rgb = srgbToLinear(texTransparent.rgb);

        if(sd.id == 8.0) texTransparent.a = 0.0;

        if(sd.id != 8.0) {
            texColor = texColor;
            texTransparent.rgb = calculateLighting();
            //texTransparent.rgb = calculateVolumetricLighting(texTransparent.rgb, pd.viewPosition[0], vec3(0.0));
            texColor = mix(texColor, texTransparent.rgb, texTransparent.a);
        }

        if(sd.id == 8.0) {
            texColor = calculateWaterRefraction();
        }
        if (sd.id == 8.0) {
            texColor = calculateWaterReflections(texColor);
        }
        if(sd.id != 8.0) {
            texColor = calculateSurfaceReflections(texColor);
        }

    }

    if(isEyeInWater == 1) {
        texColor = calculateVolumetricWaterFog(texColor, pd.viewPosition[0], vec3(0.0));;
    } else {
        texColor = calculateVolumetricLighting(texColor, pd.viewPosition[0], vec3(0.0));
    }

    if(landmask(pd.depth0)) {
        //texColor = sd.normals;
    }

    outColor.r = packUnorm2x16(encodeRGBE8(texColor).xy);
    outColor.g = packUnorm2x16(encodeRGBE8(texColor).zw);

    outLuminance = sqrt(texColor);
}