//Fragment Output
layout(location = 0) out vec4 outSpectrum;

//Uniforms
uniform vec2 viewSize, viewPixelSize;

//Fragment Inputs
in vec2 textureCoordinate;

#include "/lib/universal/universal.glsl"

#include "/lib/shared/surface/water/constants.glsl"

float random(vec2 par){
   return fract(sin(dot(par.xy,vec2(12.9898,78.233))) * 43758.5453);
}

float gaussianDistribution(vec2 seed) {
	float nrnd0 = random(seed);
	float nrnd1 = random(seed + 0.1);
	//return nrnd1 * sqrt(-2.0 * log(max(0.001, nrnd0)) / max(0.001, nrnd0));
    return sqrt(-2.0 * log(max(0.001, nrnd0))) * cos(tau*nrnd1);
}

float phillips(in vec2 k, in vec2 windDirection, in float L) {
    //Compute the Phillips spectrum.
    float kMagnitude = length(k);
    if(kMagnitude == 0.0) {
        return 0.0;
    }
    float kw = square(dot(normalize(k), square(windDirection)));
    float spectrum = abs(kw) * (exp(-1.0 / square(kMagnitude * L)) / pow(kMagnitude, 4.0));
    return A * spectrum;
}

struct jonswapParams {
    float velocity;
    float fetch;
    float gamma;
    float sigma1;
    float sigma2;
    float amplitudeMult;

    vec2 windDir;
};

#define NEWTON_ITER 8
#define HALLEY_ITER 8

float cbrt(float x)
{
	float y = sign(x) * uintBitsToFloat( floatBitsToUint( abs(x) ) / 3u + 0x2a514067u );

	for( int i = 0; i < NEWTON_ITER; ++i )
    	y = ( 2. * y + x / ( y * y ) ) * .333333333;

    for( int i = 0; i < HALLEY_ITER; ++i )
    {
    	float y3 = y * y * y;
        y *= ( y3 + 2. * x ) / ( 2. * y3 + x );
    }
    
    return y;
}

float calcWp(jonswapParams JS) {
    float velxF = JS.velocity * JS.fetch;
    float g2 = g*g;

    return 22.0 * cbrt(g2/velxF);
}

float calcAlpha(jonswapParams JS) {
    float velDivF = JS.velocity / JS.fetch;

    float tmp = velDivF * JS.velocity / g;

    return 0.076 * pow(tmp, 0.22);
}

float jonswap(in vec2 k, jonswapParams JS) {
    //Compute the JONSWAP spectrum.
    float f = length(k);

    float df = (length(k + 1e-5) - f) / 1e-5;

    f = f * 6.0;

    if(abs(f) == 0.0) {
        return 0.0;
    }

    float w = f;
    float wp = calcWp(JS);
    float alpha = calcAlpha(JS);
    float sigma = w > wp ? JS.sigma2 : JS.sigma1;
    float dw = w - wp;

    float rexp_denom = sigma * wp;
    float rexp = -pow(dw/rexp_denom, 2.0) / 2.0;
    float r = max(exp(rexp), 0.0);

    float gammaR = pow(JS.gamma, r);

	float jdistExp = -1.2 * pow(wp/w, 4.0);
	float c = alpha * g * g * pow(w, -5.0);

    return abs(dot(fNormalize(k), JS.windDir)) * (JS.amplitudeMult * (c * max(exp(jdistExp), 0.0) * gammaR)) * tau;
}

jonswapParams fillBaseSpectrum() {
    jonswapParams JS;

    JS.velocity = BASE_WIND_VELOCITY;
    JS.fetch = BASE_WAVE_FETCH;
    JS.gamma = BASE_WAVE_GAMMA;
    JS.sigma1 = BASE_WAVE_SIGMA1;
    JS.sigma2 = BASE_WAVE_SIGMA2;

    JS.amplitudeMult = BASE_WAVE_AMPLITUDE_MULT * rcp(SIDELENGTH * SIDELENGTH);

    JS.windDir = windDirection;

    return JS;
}

jonswapParams fillSwellSpectrum() {
    jonswapParams JS;

    JS.velocity = SWELL_WIND_VELOCITY;
    JS.fetch = SWELL_WAVE_FETCH;
    JS.gamma = SWELL_WAVE_GAMMA;
    JS.sigma1 = SWELL_WAVE_SIGMA1;
    JS.sigma2 = SWELL_WAVE_SIGMA2;

    JS.amplitudeMult = SWELL_WAVE_AMPLITUDE_MULT * rcp(SIDELENGTH*SIDELENGTH);

    JS.windDir = windDirection;

    return JS;
}

/* DRAWBUFFERS:7 */
void main() {
    if(gl_FragCoord.x > SIZE + 1.0 || gl_FragCoord.y > SIZE + 1.0) {
        discard;
    }

    jonswapParams base = fillBaseSpectrum();
    jonswapParams swell = fillSwellSpectrum();

    vec4 h;

    float n = (gl_FragCoord.x-1.5) - SIZE/2.0;
    float m = (gl_FragCoord.y-1.5) - SIZE/2.0;

    vec2 k = tau * vec2(n, m) / SIDELENGTH;

    float er = gaussianDistribution(gl_FragCoord.xy * 0.01);
    float ei = gaussianDistribution(gl_FragCoord.xy * 0.001);

    float pk = jonswap(k, base) + jonswap(k, swell);
    #ifndef USE_JONSWAP_SPECTRUM
        pk = phillips(k, windDirection, L);
    #endif

    h.x = RMS * er * sqrt(pk);
    h.y = RMS * ei * sqrt(pk);

    er = gaussianDistribution((viewSize+gl_FragCoord.xy) * 0.01);
    ei = gaussianDistribution((viewSize+gl_FragCoord.xy) * 0.001);

    pk = jonswap(-k, base) + jonswap(-k, swell);
    #ifndef USE_JONSWAP_SPECTRUM
        pk = phillips(-k, windDirection, L);
    #endif

    h.z = RMS * er * sqrt(pk);
    h.w = RMS * ei * sqrt(pk);

    outSpectrum = h;
}