/********************************************************
    © 2020 Continuum Graphics LLC. All Rights Reserved
 ********************************************************/

#if !defined _ACES_LMT_
#define _ACES_LMT_

/* sRGB <-> ACEScct Functions */

const float X_BRK = 0.0078125;
const float Y_BRK = 0.155251141552511;
const float A = 10.5402377416545;
const float B = 0.0729055341958355;

vec3 sRGB_to_ACEScct(vec3 color_sRGB) {
    vec3 color_ACEScct = vec3(0.0);
    for(int i = 0; i < 3; ++i) {
        color_ACEScct[i] = color_sRGB[i] <= X_BRK ? A * color_sRGB[i] + B : (log2(color_sRGB[i]) + 9.72) / 17.52;
    }
    return color_ACEScct;
}

vec3 ACEScct_to_sRGB(vec3 color_ACEScct) {
    vec3 color_sRGB = vec3(0.0);
    for(int i = 0; i < 3; ++i) {
        color_sRGB[i] = color_ACEScct[i] > Y_BRK ? exp2(color_ACEScct[i] * 17.52 - 9.72) : (color_ACEScct[i] - B) / A;
    }
    return color_sRGB;
}

/* ACES <-> ACEScct Functions */

vec3 ACES_to_ACEScct(vec3 color_ACES) {
    vec3 color_RGB = color_ACES * AP0_2_AP1;
    return sRGB_to_ACEScct(color_RGB);
}

vec3 ACEScct_to_ACES(vec3 color_ACEScct) {
    vec3 color_sRGB = ACEScct_to_sRGB(color_ACEScct);
    return color_sRGB * AP1_2_AP0;
}

/* ASC CDL LMT */

vec3 LMT_ASC_CDL(vec3 color_ACES, vec3 slope, vec3 offset, vec3 power, float saturation) {
    /*
        Slope  -> Gain
        Offset -> Lift
        Power  -> Gamma
    */

    // ACES -> ACEScct
    vec3 color_ACEScct = ACES_to_ACEScct(color_ACES);

    // Slope, Offset, Power
    color_ACEScct = pow((color_ACEScct * slope) + offset, power);

    // Saturation
    float luma = dot(color_ACEScct, vec3(0.2126, 0.7152, 0.0722));
    float satClamp = max0(saturation);
    color_ACEScct = (color_ACEScct - luma) * satClamp + luma;

    // ACEScct -> ACES
    return ACEScct_to_ACES(color_ACEScct);
}

/* Linear Gamma Adjustment LMT */

vec3 LMT_Gamma_Adjust_Linear(vec3 color_ACES, float gamma, float pivot) {
    // Treat as contrast adjustment
    const float scalar = pivot / pow(pivot, gamma);
    return pow(color_ACES, vec3(gamma)) * scalar;
}

/* Hue Rotation LMT */

float interpolate1D(vec2 table[2], float p) {
    if( p <= table[0].x ) return table[0].y;
    if( p >= table[2-1].x ) return table[2-1].y;
    
    for(int i = 0; i < 2 - 1; ++i) {
        if( table[i].x <= p && p < table[i+1].x ) {
            float s = (p - table[i].x) / (table[i+1].x - table[i].x);
            return table[i].y * ( 1 - s ) + table[i+1].y * s;
        }
    }

    return 0.0;
}

vec3 rotate_H_in_H(vec3 color_ACES, float centerH, float widthH, float degreesShift) {
    vec3 color_YCH = rgb_2_ych(color_ACES);
    vec3 color_YCHnew = color_YCH;

    float centeredHue = center_hue(color_YCH.z, centerH);
    float f_H = cubic_basis_shaper(centeredHue, widthH);

    float oldHue = centeredHue;
    float newHue = centeredHue + degreesShift;
    
    float blendedHue = mix(oldHue, newHue, f_H);

    if(f_H > 0.0) color_YCHnew.z = uncenter_hue(blendedHue, centerH);
    
    return ych_2_rgb(color_YCHnew);
}

vec3 scale_C_at_H(vec3 rgb, float centerH, float widthH, float percentC) {
    vec3 new_rgb = rgb;
    vec3 ych = rgb_2_ych( rgb);

    if (ych.y > 0.0f) {  // Only do the chroma adjustment if pixel is non-neutral
        float centeredHue = center_hue(ych.z, centerH);
        float f_H = cubic_basis_shaper(centeredHue, widthH);

        if (f_H > 0.0f) {
            // Scale chroma in affected hue region
            vec3 new_ych = ych;
            new_ych.y = ych.y * (f_H * (percentC - 1.0) + 1.0);
            new_rgb = ych_2_rgb( new_ych);
        }
    }

    return new_rgb;
}

vec3 scale_Y_at_H(vec3 rgb, float centerH, float widthH, float percentY) {
    vec3 new_rgb = rgb;
    vec3 ych = rgb_2_ych( rgb);

    if (ych.y > 0.0f) {  // Only do the chroma adjustment if pixel is non-neutral
        float centeredHue = center_hue(ych.z, centerH);
        float f_H = cubic_basis_shaper(centeredHue, widthH);

        if (f_H > 0.0f) {
            // Scale chroma in affected hue region
            vec3 new_ych = ych;
            new_ych.x = ych.x * (f_H * (percentY - 1.0) + 1.0);
            new_rgb = ych_2_rgb( new_ych);
        } 
    }

    return new_rgb;
}

vec3 scale_C(vec3 rgb, float percentC) {
    vec3 ych = rgb_2_ych( rgb);
    ych.y = ych.y * percentC;
    
    return ych_2_rgb(ych);
}

vec3 scale_Y(vec3 rgb, float percentY) {
    vec3 ych = rgb_2_ych(rgb);
    ych.x = ych.x * percentY;
    
    return ych_2_rgb(ych);
}

/* White Balance LMT */

mat3 ChromaticAdaptation(vec2 src_xy, vec2 dst_xy) {
    const mat3 ConeResponse = mat3(
         0.8951,  0.2664, -0.1614,
        -0.7502,  1.7135,  0.0367,
         0.0389, -0.0685,  1.0296
    );
    const mat3 InvConeResponse = mat3(
         0.9869929, -0.1470543, 0.1599627,
         0.4323053,  0.5183603, 0.0492912,
        -0.0085287,  0.0400428, 0.9684867
    );

    vec3 src_XYZ = xyY_2_XYZ(vec3(src_xy, 1.0));
    vec3 dst_XYZ = xyY_2_XYZ(vec3(dst_xy, 1.0));

    vec3 src_coneResp = src_XYZ * ConeResponse;
    vec3 dst_coneResp = dst_XYZ * ConeResponse;

    mat3 VonKriesMat = mat3(
        dst_coneResp[0] / src_coneResp[0], 0.0, 0.0,
        0.0, dst_coneResp[1] / src_coneResp[1], 0.0,
        0.0, 0.0, dst_coneResp[2] / src_coneResp[2]
    );

    return (ConeResponse * VonKriesMat) * InvConeResponse;
}

vec2 PlanckianLocusChromaticity(float Temp) {
    float u = (0.860117757 + 1.54118254e-4 * Temp + 1.28641212e-7 * Temp * Temp) / (1.0f + 8.42420235e-4 * Temp + 7.08145163e-7 * Temp * Temp);
    float v = (0.317398726 + 4.22806245e-5 * Temp + 4.20481691e-8 * Temp * Temp) / (1.0f - 2.89741816e-5 * Temp + 1.61456053e-7 * Temp * Temp);

    float x = 3.0 * u / (2.0 * u - 8.0 * v + 4.0);
    float y = 2.0 * v / (2.0 * u - 8.0 * v + 4.0);

    return vec2(x, y);
}

vec2 D_IlluminantChromaticity(float Temp) {
    // Accurate for 4000K < Temp < 25000K
    // in: correlated color temperature
    // out: CIE 1931 chromaticity
    // Correct for revision of Plank's law
    // This makes 6500 == D65
    Temp *= 1.000556328;

    float x = Temp <= 7000 ?
              0.244063 + ( 0.09911e3 + ( 2.9678e6 - 4.6070e9 / Temp ) / Temp ) / Temp :
              0.237040 + ( 0.24748e3 + ( 1.9018e6 - 2.0064e9 / Temp ) / Temp ) / Temp;

    float y = -3.0 * x * x + 2.87 * x - 0.275;

    return vec2(x, y);
}

vec2 PlanckianIsothermal( float Temp, float Tint ) {
    float u = (0.860117757 + 1.54118254e-4 * Temp + 1.28641212e-7 * Temp * Temp) / (1.0 + 8.42420235e-4 * Temp + 7.08145163e-7 * Temp * Temp);
    float v = (0.317398726 + 4.22806245e-5 * Temp + 4.20481691e-8 * Temp * Temp) / (1.0 - 2.89741816e-5 * Temp + 1.61456053e-7 * Temp * Temp);

    float ud = (-1.13758118e9 - 1.91615621e6 * Temp - 1.53177 * Temp * Temp) / pow(1.41213984e6 + 1189.62 * Temp + Temp * Temp, 2.0);
    float vd = ( 1.97471536e9 - 705674.0 * Temp - 308.607 * Temp * Temp) / pow(6.19363586e6 - 179.456 * Temp + Temp * Temp, 2.0); //don't pow2 this

    vec2 uvd = normalize(vec2(u, v));

    // Correlated color temperature is meaningful within +/- 0.05
    u += -uvd.y * Tint * 0.05;
    v +=  uvd.x * Tint * 0.05;

    float x = 3.0 * u / (2.0 * u - 8.0 * v + 4.0);
    float y = 2.0 * v / (2.0 * u - 8.0 * v + 4.0);

    return vec2(x, y);
}

vec3 WhiteBalance(vec3 LinearColor, const float WhiteTemp, const float WhiteTint) {
    LinearColor = LinearColor * AP0_2_AP1;

    vec2 SrcWhiteDaylight = D_IlluminantChromaticity(WhiteTemp);
    vec2 SrcWhitePlankian = PlanckianLocusChromaticity(WhiteTemp);

    vec2 SrcWhite = WhiteTemp < 4000 ? SrcWhitePlankian : SrcWhiteDaylight;
    const vec2 D65White = vec2(0.31270,  0.32900);

    // Offset along isotherm
    vec2 Isothermal = PlanckianIsothermal(WhiteTemp, WhiteTint * 0.25) - SrcWhitePlankian;
    SrcWhite += Isothermal;

    mat3x3 WhiteBalanceMat = ChromaticAdaptation(SrcWhite, D65White);
    WhiteBalanceMat = (sRGB_2_XYZ * WhiteBalanceMat) * XYZ_2_sRGB;

    return LinearColor * WhiteBalanceMat * AP1_2_AP0;
}

#endif
