/*
 * Decompiled with CFR 0.152.
 */
package net.skinsrestorer.shared.connections;

import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.Locale;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import net.skinsrestorer.api.PropertyUtils;
import net.skinsrestorer.api.connections.MineSkinAPI;
import net.skinsrestorer.api.connections.model.MineSkinResponse;
import net.skinsrestorer.api.exception.DataRequestException;
import net.skinsrestorer.api.exception.MineSkinException;
import net.skinsrestorer.api.property.SkinProperty;
import net.skinsrestorer.api.property.SkinVariant;
import net.skinsrestorer.shadow.configme.SettingsManager;
import net.skinsrestorer.shadow.javax.inject.Inject;
import net.skinsrestorer.shadow.jbannotations.Nullable;
import net.skinsrestorer.shadow.kyori.adventure.text.minimessage.tag.resolver.TagResolver;
import net.skinsrestorer.shared.config.APIConfig;
import net.skinsrestorer.shared.connections.http.HttpClient;
import net.skinsrestorer.shared.connections.http.HttpResponse;
import net.skinsrestorer.shared.connections.responses.mineskin.MineSkinErrorDelayResponse;
import net.skinsrestorer.shared.connections.responses.mineskin.MineSkinErrorResponse;
import net.skinsrestorer.shared.connections.responses.mineskin.MineSkinUrlResponse;
import net.skinsrestorer.shared.exception.DataRequestExceptionShared;
import net.skinsrestorer.shared.exception.MineSkinExceptionShared;
import net.skinsrestorer.shared.log.SRLogLevel;
import net.skinsrestorer.shared.log.SRLogger;
import net.skinsrestorer.shared.subjects.messages.Message;
import net.skinsrestorer.shared.subjects.messages.SkinsRestorerLocale;
import net.skinsrestorer.shared.utils.MetricsCounter;

public class MineSkinAPIImpl
implements MineSkinAPI {
    private static final URI MINESKIN_ENDPOINT = URI.create("https://api.mineskin.org/generate/url/");
    private static final String NAMEMC_SKIN_URL = "https://namemc.com/skin/";
    private static final String NAMEMC_IMG_URL = "https://s.namemc.com/i/%s.png";
    private final ExecutorService executorService = Executors.newSingleThreadExecutor(r -> {
        Thread t = new Thread(r);
        t.setName("SkinsRestorer-MineSkinAPI");
        return t;
    });
    private final SRLogger logger;
    private final MetricsCounter metricsCounter;
    private final SettingsManager settings;
    private final SkinsRestorerLocale locale;
    private final HttpClient httpClient;

    @Override
    public MineSkinResponse genSkin(String imageUrl, @Nullable SkinVariant skinVariant) throws DataRequestException, MineSkinException {
        String resultUrl = imageUrl.startsWith(NAMEMC_SKIN_URL) ? NAMEMC_IMG_URL.replace("%s", imageUrl.substring(24)) : imageUrl;
        AtomicInteger retryAttempts = new AtomicInteger(0);
        do {
            try {
                Optional optional = CompletableFuture.supplyAsync(() -> {
                    try {
                        return this.genSkinInternal(resultUrl, skinVariant);
                    }
                    catch (DataRequestException | MineSkinException e) {
                        throw new CompletionException(e);
                    }
                    catch (IOException e) {
                        this.logger.debug(SRLogLevel.WARNING, "[ERROR] MineSkin Failed! IOException (connection/disk): (" + resultUrl + ")", e);
                        throw new CompletionException(new DataRequestExceptionShared(e));
                    }
                    catch (InterruptedException e) {
                        throw new RuntimeException(e);
                    }
                }, this.executorService).join();
                if (optional.isPresent()) {
                    return (MineSkinResponse)optional.get();
                }
                retryAttempts.incrementAndGet();
            }
            catch (CompletionException e) {
                if (e.getCause() instanceof DataRequestException) {
                    throw new DataRequestExceptionShared(e.getCause());
                }
                if (e.getCause() instanceof MineSkinException) {
                    throw new MineSkinExceptionShared((MineSkinException)e.getCause());
                }
                throw new RuntimeException(e);
            }
        } while (retryAttempts.get() < 5);
        throw new MineSkinExceptionShared(this.locale, Message.ERROR_MS_API_FAILED, new TagResolver[0]);
    }

    private Optional<MineSkinResponse> genSkinInternal(String imageUrl, @Nullable SkinVariant skinVariant) throws DataRequestException, MineSkinException, IOException, InterruptedException {
        String skinVariantString = skinVariant != null ? "&variant=" + skinVariant.name().toLowerCase(Locale.ROOT) : "";
        HttpResponse response = this.queryURL("url=" + URLEncoder.encode(imageUrl, "UTF-8") + skinVariantString);
        this.logger.debug("MineSkinAPI: Response: " + response);
        switch (response.getStatusCode()) {
            case 200: {
                MineSkinUrlResponse urlResponse = response.getBodyAs(MineSkinUrlResponse.class);
                SkinProperty property = SkinProperty.of(urlResponse.getData().getTexture().getValue(), urlResponse.getData().getTexture().getSignature());
                return Optional.of(MineSkinResponse.of(property, urlResponse.getIdStr(), skinVariant, PropertyUtils.getSkinVariant(property)));
            }
            case 400: 
            case 500: {
                MineSkinErrorResponse errorResponse = response.getBodyAs(MineSkinErrorResponse.class);
                String error = errorResponse.getErrorCode();
                this.logger.debug(String.format("[ERROR] MineSkin Failed! Reason: %s Image URL: %s", error, imageUrl));
                switch (error) {
                    case "failed_to_create_id": 
                    case "skin_change_failed": {
                        this.logger.debug("Trying again in 5 seconds...");
                        TimeUnit.SECONDS.sleep(5L);
                        return Optional.empty();
                    }
                    case "no_account_available": {
                        throw new MineSkinExceptionShared(this.locale, Message.ERROR_MS_FULL, new TagResolver[0]);
                    }
                }
                throw new MineSkinExceptionShared(this.locale, Message.ERROR_INVALID_URLSKIN, new TagResolver[0]);
            }
            case 403: {
                MineSkinErrorResponse apiErrorResponse = response.getBodyAs(MineSkinErrorResponse.class);
                String errorCode2 = apiErrorResponse.getErrorCode();
                String error2 = apiErrorResponse.getError();
                if (errorCode2.equals("invalid_api_key")) {
                    this.logger.severe("[ERROR] MineSkin API key is not invalid! Reason: " + error2);
                    switch (error2) {
                        case "Invalid API Key": {
                            this.logger.severe("The API Key provided is not registered on MineSkin! Please empty \"api.mineSkinKey\" in plugins/SkinsRestorer/config.yml and run /sr reload");
                            break;
                        }
                        case "Client not allowed": {
                            this.logger.severe("This server ip is not on the apikey allowed IPs list!");
                            break;
                        }
                        case "Origin not allowed": {
                            this.logger.severe("This server Origin is not on the apikey allowed Origins list!");
                            break;
                        }
                        case "Agent not allowed": {
                            this.logger.severe("SkinsRestorer's agent \"SkinsRestorer/MineSkinAPI\" is not on the apikey allowed agents list!");
                            break;
                        }
                        default: {
                            this.logger.severe("Unknown error, please report this to SkinsRestorer's discord!");
                        }
                    }
                    throw new MineSkinExceptionShared(this.locale, Message.ERROR_MS_API_KEY_INVALID, new TagResolver[0]);
                }
                throw new MineSkinExceptionShared(this.locale, Message.ERROR_MS_UNKNOWN, new TagResolver[0]);
            }
            case 429: {
                MineSkinErrorDelayResponse errorDelayResponse = response.getBodyAs(MineSkinErrorDelayResponse.class);
                if (errorDelayResponse.getDelay() != null) {
                    TimeUnit.SECONDS.sleep(errorDelayResponse.getDelay().intValue());
                } else if (errorDelayResponse.getNextRequest() != null) {
                    Instant nextRequestInstant = Instant.ofEpochSecond(errorDelayResponse.getNextRequest().intValue());
                    int delay = (int)Duration.between(Instant.now(), nextRequestInstant).getSeconds();
                    if (delay > 0) {
                        TimeUnit.SECONDS.sleep(delay);
                    }
                } else {
                    TimeUnit.SECONDS.sleep(2L);
                }
                return Optional.empty();
            }
        }
        this.logger.debug("[ERROR] MineSkin Failed! Unknown error: (Image URL: " + imageUrl + ") " + response.getStatusCode());
        throw new MineSkinExceptionShared(this.locale, Message.ERROR_MS_API_FAILED, new TagResolver[0]);
    }

    private HttpResponse queryURL(String query) throws IOException {
        int i = 0;
        while (true) {
            try {
                this.metricsCounter.increment(MetricsCounter.Service.MINE_SKIN);
                HashMap<String, String> headers = new HashMap<String, String>();
                Optional<String> apiKey = this.getApiKey(this.settings);
                if (apiKey.isPresent()) {
                    headers.put("Authorization", String.format("Bearer %s", apiKey));
                }
                return this.httpClient.execute(MINESKIN_ENDPOINT, new HttpClient.RequestBody(query, HttpClient.HttpType.FORM), HttpClient.HttpType.JSON, "SkinsRestorer/MineSkinAPI", HttpClient.HttpMethod.POST, headers, 90000);
            }
            catch (IOException e) {
                if (i >= 2) {
                    throw e;
                }
                ++i;
                continue;
            }
            break;
        }
    }

    private Optional<String> getApiKey(SettingsManager settings) {
        String apiKey = settings.getProperty(APIConfig.MINESKIN_API_KEY);
        if (apiKey.isEmpty() || apiKey.equals("key")) {
            return Optional.empty();
        }
        return Optional.of(apiKey);
    }

    @Inject
    public MineSkinAPIImpl(SRLogger logger, MetricsCounter metricsCounter, SettingsManager settings, SkinsRestorerLocale locale, HttpClient httpClient) {
        this.logger = logger;
        this.metricsCounter = metricsCounter;
        this.settings = settings;
        this.locale = locale;
        this.httpClient = httpClient;
    }
}

