/*
 * Decompiled with CFR 0.152.
 */
package org.spongepowered.common.world.storage;

import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.DirectoryIteratorException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtAccounter;
import net.minecraft.nbt.NbtIo;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.spongepowered.api.Server;
import org.spongepowered.api.data.persistence.DataContainer;
import org.spongepowered.api.data.persistence.DataQuery;
import org.spongepowered.common.SpongeCommon;
import org.spongepowered.common.data.persistence.NBTTranslator;
import org.spongepowered.common.world.server.SpongeWorldManager;
import org.spongepowered.common.world.storage.SpongePlayerData;

public final class SpongePlayerDataManager {
    private static final String SPONGE_DATA = "sponge";
    private final Server server;
    private final Map<UUID, SpongePlayerData> playerDataByUniqueId;
    private @Nullable Path playersDirectory = null;

    public SpongePlayerDataManager(Server server) {
        this.server = server;
        this.playerDataByUniqueId = new ConcurrentHashMap<UUID, SpongePlayerData>();
    }

    public void load() {
        try {
            this.playersDirectory = ((SpongeWorldManager)this.server.worldManager()).getDefaultWorldDirectory().resolve("data").resolve(SPONGE_DATA);
            Files.createDirectories(this.playersDirectory, new FileAttribute[0]);
            ArrayList<Path> playerFiles = new ArrayList<Path>();
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.playersDirectory, "*.{dat}");){
                for (Path entry : stream) {
                    playerFiles.add(entry);
                }
            }
            catch (DirectoryIteratorException e) {
                SpongeCommon.logger().error("Something happened when trying to gather all player files", (Throwable)e);
            }
            for (Path playerFile : playerFiles) {
                CompoundTag compound;
                if (!Files.isReadable(playerFile)) continue;
                try (InputStream stream = Files.newInputStream(playerFile, new OpenOption[0]);){
                    compound = NbtIo.readCompressed((InputStream)stream, (NbtAccounter)NbtAccounter.unlimitedHeap());
                }
                catch (Exception e) {
                    throw new RuntimeException("Failed to decompress playerdata for playerfile " + String.valueOf(playerFile), e);
                }
                if (compound.isEmpty()) {
                    throw new RuntimeException("Failed to decompress player data within [" + String.valueOf(playerFile) + "]!");
                }
                DataContainer container = NBTTranslator.INSTANCE.translateFrom(compound);
                SpongePlayerData data = container.getSerializable(DataQuery.of(), SpongePlayerData.class).get();
                this.playerDataByUniqueId.put(data.getUniqueId(), data);
            }
            playerFiles.clear();
        }
        catch (Exception ex) {
            throw new RuntimeException("Encountered an exception while creating the player data handler!", ex);
        }
    }

    public void readPlayerData(CompoundTag compound, @Nullable UUID playerUniqueId, @Nullable Instant creation) {
        if (creation == null) {
            creation = Instant.now();
        }
        Instant lastPlayed = Instant.now();
        if (compound.contains("bukkit", 10)) {
            CompoundTag bukkitCompound = compound.getCompound("bukkit");
            creation = Instant.ofEpochMilli(bukkitCompound.getLong("firstPlayed"));
            lastPlayed = Instant.ofEpochMilli(bukkitCompound.getLong("lastPlayed"));
        }
        if (compound.contains("Canary", 10)) {
            CompoundTag canaryCompound = compound.getCompound("Canary");
            creation = Instant.ofEpochMilli(canaryCompound.getLong("FirstJoin"));
            lastPlayed = Instant.ofEpochMilli(canaryCompound.getLong("LastJoin"));
        }
        if (playerUniqueId == null && compound.hasUUID("UUID")) {
            playerUniqueId = compound.getUUID("UUID");
        }
        if (playerUniqueId != null) {
            Optional<Instant> savedJoined;
            Optional<Instant> savedFirst = this.getFirstJoined(playerUniqueId);
            if (savedFirst.isPresent()) {
                creation = savedFirst.get();
            }
            if ((savedJoined = this.getLastPlayed(playerUniqueId)).isPresent()) {
                lastPlayed = savedJoined.get();
            }
            this.setPlayerInfo(playerUniqueId, creation, lastPlayed);
        }
    }

    public void saveSpongePlayerData(UUID uniqueId) {
        if (uniqueId == null) {
            throw new IllegalArgumentException("Player unique id cannot be null!");
        }
        @Nullable SpongePlayerData data = this.playerDataByUniqueId.get(uniqueId);
        if (data != null) {
            this.saveFile(uniqueId.toString(), this.createCompoundFor(data));
        } else {
            SpongeCommon.logger().error("Couldn't find a player data for the uuid: " + uniqueId.toString());
        }
    }

    private CompoundTag createCompoundFor(SpongePlayerData data) {
        return NBTTranslator.INSTANCE.translate(data.toContainer());
    }

    private void saveFile(String id, CompoundTag compound) {
        try {
            Files.createDirectories(this.playersDirectory, new FileAttribute[0]);
            Path finalDatPath = this.playersDirectory.resolve(id + ".dat");
            Path newDatPath = this.playersDirectory.resolve(id + ".dat.tmp");
            try (OutputStream stream = Files.newOutputStream(newDatPath, StandardOpenOption.CREATE);){
                NbtIo.writeCompressed((CompoundTag)compound, (OutputStream)stream);
            }
            Files.move(newDatPath, finalDatPath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (Exception e) {
            SpongeCommon.logger().error("Failed to save player data for [{}]!", (Object)id, (Object)e);
        }
    }

    public void setPlayerInfo(UUID playerUniqueId, Instant join, Instant last) {
        if (playerUniqueId == null) {
            throw new IllegalArgumentException("Player unique id cannot be null!");
        }
        if (join == null) {
            throw new IllegalArgumentException("Joined date cannot be null!");
        }
        if (last == null) {
            throw new IllegalArgumentException("Last joined date cannot be null!");
        }
        SpongePlayerData data = this.playerDataByUniqueId.get(playerUniqueId);
        if (data == null) {
            data = new SpongePlayerData();
            data.setUniqueId(playerUniqueId);
        }
        data.setFirstJoined(join.toEpochMilli());
        data.setLastJoined(last.toEpochMilli());
        this.playerDataByUniqueId.put(playerUniqueId, data);
    }

    public Optional<Instant> getFirstJoined(UUID uniqueId) {
        SpongePlayerData data = this.playerDataByUniqueId.get(uniqueId);
        return Optional.ofNullable(data == null ? null : Instant.ofEpochMilli(data.getFirstJoined()));
    }

    public Optional<Instant> getLastPlayed(UUID uniqueId) {
        SpongePlayerData data = this.playerDataByUniqueId.get(uniqueId);
        return Optional.ofNullable(data == null ? null : Instant.ofEpochMilli(data.getLastJoined()));
    }
}

