//Fragment Output
layout(location = 0) out vec4 outColor;
layout(location = 1) out vec4 outSky;

//Samplers
uniform usampler2D colortex2;
uniform usampler2D colortex3;
uniform sampler2D colortex5;
uniform sampler2D depthtex0;

uniform sampler2D gaux4;

uniform sampler2D noisetex;

uniform sampler2D shadowtex0;
uniform sampler2D shadowtex1;
uniform sampler2D shadowcolor0;
uniform sampler2D shadowcolor1;

//Sky Textures
uniform sampler2D depthtex1;
uniform sampler2D depthtex2;

//Uniforms
uniform mat4 gbufferProjection, gbufferModelView;
uniform mat4 gbufferProjectionInverse, gbufferModelViewInverse;

uniform mat4 shadowProjection, shadowModelView;
uniform mat4 shadowProjectionInverse, shadowModelViewInverse;

uniform vec3 sunVector, moonVector, shadowLightVector, upVector;
uniform vec3 sunVectorView, moonVectorView, shadowLightVectorView;
uniform vec3 cameraPosition;

uniform vec2 viewSize, viewPixelSize;

uniform ivec2 eyeBrightness;

uniform float eyeAltitude, frameTimeCounter;

uniform int frameCounter, isEyeInWater;

//Fragment Inputs
struct lightStruct {
    vec3 indirect;
    vec3 direct;

    vec3 skySh[9];
};

in lightStruct lc;
in vec2 textureCoordinate;

#include "/lib/universal/universal.glsl"
#include "/lib/fragment/dither/bayer.glsl"
#include "/lib/fragment/dither/hash.glsl"
#include "/lib/fragment/dither/hammersley.glsl"
#include "/lib/fragment/colorspace.glsl"

#include "/lib/fragment/lighting/direct/distortion.glsl"
#include "/lib/fragment/lighting/direct/hard.glsl"
#include "/lib/fragment/lighting/direct/soft.glsl"

#include "/lib/ray/rayShape.glsl"

#include "/lib/shared/sky/celestial/constants.glsl"

#include "/lib/shared/sky/constants.glsl"
#include "/lib/shared/sky/phase.glsl"
#include "/lib/shared/sky/atmosphere/lookup.glsl"
#include "/lib/shared/sky/atmosphere/scattering.glsl"
#include "/lib/shared/sky/atmosphere/transmittance.glsl"
#include "/lib/shared/sky/projection.glsl"

#include "/lib/shared/surface/water/constants.glsl"
#include "/lib/shared/surface/water/waves.glsl"

#include "/lib/shared/sky/clouds/volumetric/constants.glsl"
#include "/lib/shared/sky/clouds/volumetric/functions.glsl"
#include "/lib/shared/sky/clouds/volumetric/volumetric.glsl"
#include "/lib/shared/sky/clouds/volumetric/shadow.glsl"

struct positionDepth {
    float depth0;
    float depth1;

    vec3 screenPosition;
    vec3 viewPosition;
    vec3 scenePosition;
} pd;

void fillPositionStruct() {
    vec2 coord = (textureCoordinate - 0.5) * 2.5;
    pd.depth0 = texture(depthtex0, coord).r;
    pd.depth1 = texture(depthtex1, coord).r;

    pd.screenPosition = vec3(coord, pd.depth0);
    pd.viewPosition = screenSpaceToViewSpace(pd.screenPosition, gbufferProjectionInverse);
    pd.scenePosition = viewSpaceToSceneSpace(pd.viewPosition, gbufferModelViewInverse);
}

vec2 calculateDitherPattern() {
    vec2 coordinate = textureCoordinate;

    coordinate *= viewSize;
    coordinate = mod(coordinate, vec2(2.0));
    coordinate /= noiseTextureResolution;

    return texture(noisetex, coordinate).rg;
}

float calculateCausticsNV(in vec3 worldPosition, in float depth) {
	float maxSamples = CAUSTICS_SAMPLES;
    float samples = maxSamples;
    float focus = 0.8;
    vec3 lightVector = -shadowLightVector;
    float surfaceDistanceUp = depth * abs(lightVector.y);

    float filterRadius = 0.1;

    float radius = filterRadius * depth;
    float inverseDistanceThreshold = sqrt(samples / pi) * focus / radius;

    vec3 flatRefractVector = refract(lightVector, vec3(0, 1, 0), 1.00029/1.333);
    vec3 flatRefract = flatRefractVector * surfaceDistanceUp / abs(flatRefractVector.y);
    vec3 surfacePosition = worldPosition - flatRefract;

    vec2 dither = calculateDitherPattern();

    float finalCaustic = 0.0;
    for(float i = 0; i <= samples; ++i) {
        vec3 samplePos = surfacePosition;
        samplePos.xz += circleMap(i + dither.x, samples) * radius;
        vec3 refractVector = refract(lightVector, fNormalize(waterNormal(samplePos.xz)), 1.00029/1.333);

        samplePos = refractVector * (surfaceDistanceUp / abs(refractVector.y)) + samplePos;

        finalCaustic += saturate(1.0 - distance(worldPosition, samplePos) * inverseDistanceThreshold);
    }
    finalCaustic *= square(focus);
    return pow(finalCaustic, CAUSTICS_CONTRAST);
}

float calculateCausticsSEUS(in vec3 worldPosition) {
    /* 
        SEUS style caustics, these are less noisy than my own but take a lot more samples to converge on an accurate result.
    */
    vec3 lightVector = -shadowLightVector;
    float surfaceDistanceUp = min(abs(worldPosition.y - 63.0), 2.0);

    vec3 flatRefractVector = refract(lightVector, vec3(0, 1, 0), 1.00029/1.333);
    float flatRefract = surfaceDistanceUp / -flatRefractVector.y;
    vec3 surfacePosition = worldPosition - flatRefractVector * flatRefract;

    vec2 dither = calculateDitherPattern();

    cFloat radius = 0.2;
    cFloat distanceThreshold = 0.25;

    cInt numSamples = 3;
    int c = 0;

    float caustics = 0.0;
    for(int x = -numSamples; x <= numSamples; ++x) {
        for(int y = -numSamples; y <= numSamples; ++y) {
            vec2 offset = vec2(x + dither.x, y + dither.y) * radius;
            vec3 samplePosition = surfacePosition + vec3(offset.x, 0.0, offset.y);
            vec3 refractVector = refract(lightVector, fNormalize(waterNormal(samplePosition.xz)), 1.00029/1.333);
            float rayLength = surfaceDistanceUp / refractVector.y;

            vec3 collisionPoint = samplePosition - refractVector * rayLength;

            float dist = distance(collisionPoint, worldPosition);

            caustics += 1.0 - saturate(dist / distanceThreshold);

            c++;
        }
    }
    caustics /= c;
    caustics /= distanceThreshold;

    return square(caustics) * 64.0;
}

float calculateCloudShadowMap(in vec2 coordinate) {
    vec3 pos = vec3(coordinate, 0.0);
    pos.xy /= 256 * viewPixelSize;
    if (saturate(pos.xy) != pos.xy) { return 1.0; }
    pos.xy  = pos.xy * 2.0 - 1.0;
    pos.xy /= 1.0 - length(pos.xy);
    pos.xy *= 200.0;
    pos     = mat3(shadowModelViewInverse) * pos;

    pos += cameraPosition;
    pos += shadowLightVector * (256.0 - pos.y) / shadowLightVector.y;

    return calculateCloudShadow(pos, shadowLightVector, 50.0);
}

float calculateCausticShadowMap(in vec2 coordinate, in int size, in float depth) {
    #ifdef CAUSTICS
        vec3 pos = vec3(coordinate, 0.0);
        pos.xy /= size * viewPixelSize;
        if (saturate(pos.xy) != pos.xy) { return 0.0; }
        pos.xy  = pos.xy * 2.0 - 1.0;
        pos.xy /= 1.0 - length(pos.xy);
        pos.xy *= 12.0;
        pos     = mat3(shadowModelViewInverse) * pos;

        pos += cameraPosition;
        pos += shadowLightVector * (1.0 - pos.y) / shadowLightVector.y;

        #ifdef SEUS_V11_CAUSTICS
            return calculateCausticsSEUS(pos);
        #else
            return calculateCausticsNV(pos, depth);
        #endif
    #else
        return 1.0;
    #endif
}

float calculateCausticHandShadowMap(in vec2 coordinate) {
    /*
        This renders the caustics as viewed from the camera.
        Currently not used.
    */
    return 1.0;
}

/* DRAWBUFFERS:65 */
void main() {
    fillPositionStruct();

    if(textureCoordinate.x > .5 && textureCoordinate.y > .5) {
        float cloudDistance;
        outColor = calculateVolumetricClouds(vec3(0.0, planetRadius, 0.0), fNormalize(pd.scenePosition), fract(fract(frameCounter * (1.0 / phi)) + bayer128(gl_FragCoord.st)), cloudDistance);
        outColor = sqrt(outColor);

        vec2 planetSphere = RSI(vec3(0.0, planetRadius, 0.0), fNormalize(pd.scenePosition), planetRadius + -1.0);

        #ifdef VOLUMETRIC_CLOUDS
            outSky.a = planetSphere.y <= 0.0 ? cloudDistance : 100000.0;
        #endif
    } else if (textureCoordinate.x < .5 && textureCoordinate.y < .5) {
        outColor = vec4(0.0, 0.0, 0.0, 1.0);
        outSky.a = 0.0;
    }

    if(textureCoordinate.x < exp2(-SKY_RENDER_LOD) && textureCoordinate.y < exp2(-SKY_RENDER_LOD)) {
        vec3 cloudDirection = unprojectSky(textureCoordinate);

        float cloudDistance;
        vec4 clouds = calculateVolumetricClouds(vec3(0.0, planetRadius, 0.0), cloudDirection, fract(bayer128(gl_FragCoord.st)), cloudDistance);

        vec3 transmittanceView = atmosphereTransmittance(vec3(0.0, planetRadius, 0.0), cloudDirection);
        vec3 transmittanceFromCloud = atmosphereTransmittance(vec3(0.0, planetRadius, 0.0) + cloudDirection * cloudDistance, cloudDirection);
        vec3 transmittanceToCloud = min(transmittanceView / transmittanceFromCloud, 1.0);

        outSky.rgb = texture(colortex5, textureCoordinate).rgb;
        outSky.rgb = mix(outSky.rgb, outSky.rgb * clouds.a + clouds.rgb, square(transmittanceToCloud));
    } else if (textureCoordinate.x > exp2(-SKY_RENDER_LOD) && textureCoordinate.y > exp2(-SKY_RENDER_LOD)) {
        outSky.rgb = vec3(0.0);
    }

    if(textureCoordinate.x < .5 && textureCoordinate.y < .5) {
        outColor.a = calculateCloudShadowMap(textureCoordinate);
    }

    if(textureCoordinate.x < 0.5) {
        outColor.r = calculateCausticShadowMap(textureCoordinate, 480, 2.0);
        outColor.r += calculateCausticShadowMap(textureCoordinate-vec2(0.0, 0.5), 480/2, 10.0);
    }
}