/********************************************************
    © 2020 Continuum Graphics LLC. All Rights Reserved
 ********************************************************/

#if !defined _ACES_
#define _ACES_

#include "/../ContinuumLib/Fragment/ACES/Utility.fsh"
#include "/../ContinuumLib/Fragment/ACES/Tonescales.fsh"

/* IDT Functions */

vec3 sRGB_to_ACES(vec3 color_sRGB) {
    return color_sRGB * sRGB_2_ACES;
}

/* LMT Functions */

#include "/../ContinuumLib/Fragment/ACES/LMT.fsh"

/* RRT Functions */

#define RRT_GLOW_GAIN 0.05
#define RRT_GLOW_MID  0.08

#define RRT_RED_PIVOT 0.03
#define RRT_RED_HUE   0.0
#define RRT_RED_WIDTH 135.0
#define RRT_RED_SCALE 0.82

#define RRT_SAT_FACTOR 0.96

float glow_fwd(float ycIn, float glowGainIn, float glowMid) {
    float glowGainOut;

    if(ycIn <= 2.0 / 3.0 * glowMid) {
        glowGainOut = glowGainIn;
    } else if(ycIn >= 2.0 * glowMid) {
        glowGainOut = 0.0;
    } else {
        glowGainOut = glowGainIn * (glowMid / ycIn - 1.0 * 0.5);
    }
    
    return glowGainOut;
}

float sigmoid_shaper(float x) {
    float t = max(1.0 - abs(x * 0.5), 0.0);
    float y = 1.0 + sign(x) * (1.0 - t * t);
    return y * 0.5;
}

vec3 RRT(vec3 color_ACES) {
    // Glow
    float saturation = rgb_2_saturation(color_ACES);
    float ycIn = rgb_2_yc(color_ACES, 1.75);
    float s = sigmoid_shaper((saturation - 0.4) * 5.0);
    float addedGlow = 1.0 + glow_fwd(ycIn, RRT_GLOW_GAIN * s, RRT_GLOW_MID);

    color_ACES *= addedGlow;

    // Red Mod
    float hue = rgb_2_hue(color_ACES);
    float centeredHue = center_hue(hue, RRT_RED_HUE);
    float hueWeight = cubic_basis_shaper(centeredHue, RRT_RED_WIDTH);

    color_ACES.r += hueWeight * saturation * (RRT_RED_PIVOT - color_ACES.r) * (1.0 - RRT_RED_SCALE);

    // ACES -> RGB
    color_ACES = max(color_ACES, 0.0);
    vec3 color_RGBpre = color_ACES * AP0_2_AP1;
    color_RGBpre = max(color_RGBpre, 0.0);

    // Global Desat
    color_RGBpre = color_RGBpre * calc_sat_adjust_matrix(RRT_SAT_FACTOR, AP1_RGB2Y);

    // Apply tonescale independently in RGB
    vec3 color_RGBpost;
    color_RGBpost.r = segmented_spline_c5_fwd(color_RGBpre.r, RRT_PARAMS);
    color_RGBpost.g = segmented_spline_c5_fwd(color_RGBpre.g, RRT_PARAMS);
    color_RGBpost.b = segmented_spline_c5_fwd(color_RGBpre.b, RRT_PARAMS);

    // RGB -> OCES
    vec3 color_OCES = color_RGBpost * AP1_2_AP0;

    return color_OCES;
}

/* ODT Functions */

#include "/../ContinuumLib/Fragment/ACES/ODT.fsh"

#endif
