/********************************************************
    © 2020 Continuum Graphics LLC. All Rights Reserved
 ********************************************************/

#if !defined _ACES_TONESCALES_
#define _ACES_TONESCALES_

const mat3 M = mat3(
     0.5, -1.0,  0.5,
    -1.0,  1.0,  0.5,
     0.5,  0.0,  0.0
);

/* Spline Structs */

struct SegmentedSplineParams_c5 {
    float coeffsLow[6];     // Coeffs for B-spline between minPoint and midPoint (units of log luminance)
    float coeffsHigh[6];    // Coeffs for B-spline between midPoint and maxPoint (units of log luminance)
    vec2 minPoint;          // {luminance, luminance} Linear extension below this
    vec2 midPoint;          // {luminance, luminance} Linear
    vec2 maxPoint;          // {luminance, luminance} Linear extension above this
    float slopeLow;         // Log-log slope of low linear extension
    float slopeHigh;        // Log-log slope of high linear extension
};

struct SegmentedSplineParams_c9 {
    float coeffsLow[10];    // Coeffs for B-spline between minPoint and midPoint (units of log luminance)
    float coeffsHigh[10];   // Coeffs for B-spline between midPoint and maxPoint (units of log luminance)
    vec2 minPoint;          // {luminance, luminance} Linear extension below this
    vec2 midPoint;          // {luminance, luminance} Linear
    vec2 maxPoint;          // {luminance, luminance} Linear extension above this
    float slopeLow;         // Log-log slope of low linear extension
    float slopeHigh;        // Log-log slope of high linear extension
};

/* 5-component Segmented Spline Functions */

const SegmentedSplineParams_c5 RRT_PARAMS = SegmentedSplineParams_c5(
    float[6] ( -4.0000000000, -4.0000000000, -3.1573765773, -0.4852499958, 1.8477324706, 1.8477324706 ),    // coeffsLow
    float[6] ( -0.7185482425, 2.0810307172, 3.6681241237, 4.0000000000, 4.0000000000, 4.0000000000 ),       // coeffsHigh
    vec2(0.18 * exp2(-15.0), 0.0001),   // minPoint
    vec2(0.18, 4.8),                    // midPoint
    vec2(0.18 * exp2( 18.0), 10000.0),  // maxPoint
    0.0,    // slopeLow
    0.0     // slopeHigh
);

float segmented_spline_c5_fwd(float x, SegmentedSplineParams_c5 params) { // params should default to RRT_PARAMS
    const int N_KNOTS_LOW  = 4;
    const int N_KNOTS_HIGH = 4;

    float logMinPoint = log10(params.minPoint.x);
    float logMidPoint = log10(params.midPoint.x);
    float logMaxPoint = log10(params.maxPoint.x);

    float logx = log10(x <= 0 ? exp2(-14.0) : x);
    float logy;
    
    if(logx <= logMinPoint) {
        logy = logx * params.slopeLow + (log10(params.minPoint.y) - params.slopeLow * logMinPoint);
    } else if((logx > logMinPoint) && (logx < logMidPoint)) {
        float knot_coord = (N_KNOTS_LOW - 1) * (logx - logMinPoint) / (logMidPoint - logMinPoint);
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(params.coeffsLow[j], params.coeffsLow[j + 1], params.coeffsLow[j + 2]);

        vec3 monomials = vec3(t * t, t, 1.0);
        logy = dot(monomials, M * cf);
    } else if((logx >= logMidPoint) && (logx < logMaxPoint)) {
        float knot_coord = (N_KNOTS_HIGH - 1) * (logx - logMidPoint) / (logMaxPoint - logMidPoint);
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(params.coeffsHigh[j], params.coeffsHigh[j + 1], params.coeffsHigh[j + 2]);

        vec3 monomials = vec3(t * t, t, 1.0);
        logy = dot(monomials, M * cf);
    } else {
        logy = logx * params.slopeHigh + (log10(params.maxPoint.y) - params.slopeHigh * logMaxPoint);
    }

    return pow(10.0, logy);
}

/* 9-component Segmented Spline Functions */

const SegmentedSplineParams_c9 ODT_48nits = SegmentedSplineParams_c9(
    float[10] ( -1.6989700043, -1.6989700043, -1.4779000000, -1.2291000000, -0.8648000000, -0.4480000000, 0.0051800000, 0.4511080334, 0.9113744414, 0.9113744414 ),
    float[10] ( 0.5154386965, 0.8470437783, 1.1358000000, 1.3802000000, 1.5197000000, 1.5985000000, 1.6467000000, 1.6746091357, 1.6878733390, 1.6878733390 ),
    vec2(0.18 * exp2(-6.5), 0.02),  // minPoint
    vec2(0.18, 4.8),                // midPoint
    vec2(0.18 * exp2( 6.5), 48.0),  // maxPoint
    0.0,  // slopeLow
    0.04  // slopeHigh
);

float segmented_spline_c9_fwd(float x, SegmentedSplineParams_c9 params) { // params should default to ODT_48nits
    const int N_KNOTS_LOW  = 8;
    const int N_KNOTS_HIGH = 8;

    float logMinPoint = log10(params.minPoint.x);
    float logMidPoint = log10(params.midPoint.x);
    float logMaxPoint = log10(params.maxPoint.x);

    float logx = log10(x <= 0 ? exp2(-14.0) : x);
    float logy;

    if ( logx <= logMinPoint ) {
        logy = logx * params.slopeLow + (log10(params.minPoint.y) - params.slopeLow * logMinPoint);
    } else if (( logx > logMinPoint ) && ( logx < logMidPoint )) {
        float knot_coord = (N_KNOTS_LOW - 1) * (logx - logMinPoint) / (logMidPoint - logMinPoint);
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(params.coeffsLow[j], params.coeffsLow[j + 1], params.coeffsLow[j + 2]);
        
        vec3 monomials = vec3(t * t, t, 1.0);
        logy = dot(monomials, M * cf);
    } else if (( logx >= logMidPoint ) && ( logx < logMaxPoint )) {
        float knot_coord = (N_KNOTS_HIGH - 1) * (logx - logMidPoint) / (logMaxPoint - logMidPoint);
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(params.coeffsHigh[j], params.coeffsHigh[j + 1], params.coeffsHigh[j + 2]);
        
        vec3 monomials = vec3(t * t, t, 1.0);
        logy = dot(monomials, M * cf);
    } else { //if ( logIn >= logMaxPoint ) {
        logy = logx * params.slopeHigh + (log10(params.maxPoint.y) - params.slopeHigh * logMaxPoint);
    }

    return pow(10.0, logy);
}

#endif
