/********************************************************
    © 2020 Continuum Graphics LLC. All Rights Reserved
 ********************************************************/

#if !defined _AMBIENTOCCLUSION_
#define _AMBIENTOCCLUSION_

#define FALLOFF_START2	0.16
#define FALLOFF_END2	4.0

#define RADIUS 2.0

float Falloff(float dist2, float cosh) {
    return 2.0 * clamp01((dist2 - FALLOFF_START2) / (FALLOFF_END2 - FALLOFF_START2));
}

float GTAONoise(vec2 position) {
    return fract(52.9829189 * fract(dot(position, vec2( 0.06711056, 0.00583715))));
}

vec3 CalculateAOPosition(vec2 coord) {
    vec3 screenPos = vec3(coord, texture(depthtex1, coord).x) * 2.0 - 1.0;

    return projMAD(projInverseMatrix, screenPos) / (screenPos.z * projInverseMatrix[2].w + projInverseMatrix[3].w);
}

float CalculateGTAO(vec2 coord, vec3 position, vec3 normal, vec3 viewVector, float noise) {
    #ifndef AO
        return 1.0;
    #endif

    vec3 dir, ws, s;

    float noise2 = GTAONoise(gl_FragCoord.xy);

    vec2 offset;
    vec2 horizons = vec2(-1.0, -1.0);

    float radius = (RADIUS * (viewDimensions.x * gbufferProjection[2].z * 0.5)) / position.z;
        radius = max(AO_NUM_STEPS, radius);

    float stepsize	= radius / AO_NUM_STEPS;
    float phi		= 0.0;
    float ao		= 0.0;
    float division	= noise2 * stepsize;
    float currstep	= 1.0; // * params.y
    float dist2, invdist, falloff, cosh;

    for (int k = 0; k < AO_NUM_DIRECTIONS; ++k) {
        phi = float(k + noise) * (PI / AO_NUM_DIRECTIONS);
        currstep = 1.0 + division;

        dir = vec3(cos(phi), sin(phi), 0.0);
        horizons = vec2(-1.0);

        // calculate horizon angles
        for (int j = 0; j < AO_NUM_STEPS; ++j) {
            offset = dir.xy * currstep * pixelSize;

            // h1
            s = CalculateAOPosition(coord + offset);
            ws = s - position;

            dist2 = dot(ws, ws);
            invdist = inversesqrt(dist2);
            cosh = invdist * dot(ws, viewVector);

            falloff = Falloff(dist2, cosh);
            horizons.x = max(horizons.x, cosh - falloff);

            // h2
            s = CalculateAOPosition(coord - offset);
            ws = s - position;

            dist2 = dot(ws, ws);
            invdist = inversesqrt(dist2);
            cosh = invdist * dot(ws, viewVector);

            falloff = Falloff(dist2, cosh);
            horizons.y = max(horizons.y, cosh - falloff);

            // increment
            currstep += stepsize;
        }

        horizons = facos(horizons);

        // calculate gamma
        vec3 bitangent	= normalize(cross(dir, viewVector));
        vec3 tangent	= cross(viewVector, bitangent);
        vec3 nx			= normal - bitangent * dot(normal, bitangent);

        float nnx		= length(nx);
        float invnnx	= 1.0 / (nnx + 1e-6);			// to avoid division with zero
        float cosxi		= dot(nx, tangent) * invnnx;	// xi = gamma + HPI
        float gamma		= facos(cosxi) - HPI;
        float cosgamma	= dot(nx, viewVector) * invnnx;
        float singamma2	= -2.0 * cosxi;					// cos(x + HPI) = -sin(x)

        // clamp to normal hemisphere
        horizons.x = gamma + max(-horizons.x - gamma, -HPI);
        horizons.y = gamma + min(horizons.y - gamma, HPI);

        vec2 arc = horizons * singamma2 + cosgamma - cos(2.0 * horizons - gamma);

        // Riemann integral is additive
        ao += nnx * 0.25 * (arc.x + arc.y);
    }

    ao = ao / float(AO_NUM_DIRECTIONS);

    return ao;
}
#endif
