/********************************************************
    © 2020 Continuum Graphics LLC. All Rights Reserved
 ********************************************************/

#if !defined _VOLUMETRICCLOUDS_
#define _VOLUMETRICCLOUDS_

const float cloudMultiScattCoeff_a = 0.7;
const float cloudMultiScattCoeff_b = 0.5;
const float cloudMultiScattCoeff_c = 0.8;

float Calculate3DNoise(vec3 position){
    vec3 p = floor(position);
    vec3 b = cubeSmooth(fract(position));

    vec2 uv = 17.0 * p.z + p.xy + b.xy;
    vec2 rg = texture(noisetex, (uv + 0.5) * rNoiseTexRes).xy;

    return mix(rg.x, rg.y, b.z);
}

// Calculate cloud noise using FBM.
float CalculateCloudFBM(vec3 position, vec3 windDirection, const int octaves){
    const float octAlpha = 0.5; // The ratio of visibility between successive octaves
    const float octScale = 3.0; // The downscaling factor between successive octaves
    const float octShift = (octAlpha / octScale) / octaves; // Shift the FBM brightness based on how many octaves are active

    float accum = 0.0;
    float alpha = 0.5;
    vec3  shift = -windDirection * 1.3;
    
	position += windDirection;
    
    for (int i = 0; i < octaves; ++i) {
        accum += alpha * Calculate3DNoise(position);
        position = position * octScale + shift;
        alpha *= octAlpha;
    }
    
    return accum + octShift;
}

// Calculate cloud optical depth.
float CalculateCloudOD(vec3 position, const int octaves){
    // Early out.
    //if (position.y > volumetric_cloudMaxHeight || position.y < volumetric_cloudMinHeight) return 0.0;
    
    float wind          = TIME * -0.025;
    vec3  windDirection = vec3(wind, 0.0, wind);
    vec3  cloudPos      = position * 0.00045;
    
    float clouds = CalculateCloudFBM(cloudPos, windDirection, octaves);

    // Doing this early out is pointless since an fmb can rarely ever reach 0.0
	//if (clouds <= 0.0) return 0.0;
    
    float localCoverage = 1.0;

    #ifdef VC_LOCAL_COVERAGE
        localCoverage = texture2D(noisetex, (TIME * 50.0 + position.xz) * 0.000001).x;
        localCoverage = clamp01(localCoverage * 3.0 - 0.75) * 0.5 + 0.5;
    #endif

    float worldHeight       = position.y - volumetric_cloudMinHeight;
    float normalizedHeight  = clamp01(worldHeight / volumetric_cloudThickness);
    float heightAttenuation = Remap(normalizedHeight, 0.0, 0.2, 0.0, 1.0) * Remap(normalizedHeight, 0.8, 1.0, 1.0, 0.0);

    // Calculate the final cloudshape.
    clouds  = clouds * heightAttenuation * localCoverage * 2.0 * volumetric_cloudCoverage - (0.9 * heightAttenuation + normalizedHeight * 0.5 + 0.1);
    clouds  = clamp01(clouds * clamp01(pow2(normalizedHeight) * 3.5)) * volumetric_cloudDensity;

    return clouds;
}

float PhaseG(float cosTheta, const float g){
    float gg = g * g;
    return rPI * (gg * -0.25 + 0.25) * pow(-2.0 * (g * cosTheta) + (gg + 1.0), -1.5);
}

float CalculateCloudPhase(float vDotL){
    const float mixer = 0.5;

    float g1 = PhaseG(vDotL, 0.8);
    float g2 = PhaseG(vDotL, -0.5);

    return mix(g2, g1, mixer);
}

void CalculateMultipleScatteringCloudPhases(float vDotL, inout float phases[VC_MULTISCAT_QUALITY]){
    float cn = 1.0;

    for (int i = 0; i < VC_MULTISCAT_QUALITY; ++i){
        phases[i] = CalculateCloudPhase(vDotL * cn);

        cn *= cloudMultiScattCoeff_c;
    }
}

float CalculatePowderEffect(float od){
    return 1.0 - exp(-od * 2.0);
}

float CalculatePowderEffect(float powder, float vDotL){
    return mix(powder, 1.0, vDotL * 0.5 + 0.5);
}

float CalculateCloudTransmitDepth(vec3 position, vec3 direction, const int steps){
    float rayLength = 25.0; // Starting ray length

    float od = 0.0;

    for (int i = 0; i < steps; ++i, position += direction * rayLength){
		// When the ray goes above of the cloud boundary we make sure to break out of the loop.
		if (position.y >= volumetric_cloudMaxHeight) break;
		
        od += CalculateCloudOD(position, VC_NOISE_OCTAVES) * rayLength;

        rayLength *= 1.5;
    }

    return od;
}

float CalculateCloudTransmitDepthSky(vec3 position){
    return min(volumetric_cloudMaxHeight - position.y, volumetric_cloudMinHeight) * (1.0 / volumetric_cloudThicknessMult) * 0.0007;
}

// Calculate the total energy of the clouds.
void CalculateCloudScattering(float scatterCoeff, float transmit, float an, float bn, float transmitDepth, float transmitDepthSky, float transmitDepthGround, float phase, float powder, inout float directScattering, inout float skylightScattering, inout float groundLightScattering, const int dlSteps){
    // Absorb sunlight through the clouds.
    float absorb        = exp2(-transmitDepth    * rLOG2 * bn);
    float absorbSky     = exp2(-transmitDepthSky * rLOG2 * bn);
    #ifdef VC_GROUND_LIGHTING
        float absorbGround  = exp2(-transmitDepthGround * rLOG2 * bn);
    #endif
    
    directScattering   += scatterCoeff * transmit * an * absorb * powder * phase;
    skylightScattering += scatterCoeff * transmit * an * absorbSky;
    #ifdef VC_GROUND_LIGHTING
        groundLightScattering += scatterCoeff * transmit * an * absorbGround;
    #endif
}

void CalculateCloudScattering(vec3 position, vec3 wLightVector, float od, float vDotL, float phases[VC_MULTISCAT_QUALITY], float transmit, float stepTransmit, inout float directScattering, inout float skylightScattering, inout float groundLightScattering, const int dlSteps, const int msSteps){	
	// Scattering intergral.
    float scatterCoeff = 1.0 - stepTransmit;

    // Depth for directional transmit
    float transmitDepth    = CalculateCloudTransmitDepth(position, wLightVector, dlSteps);

    #ifdef VC_HQ_SKYLIGHTING
        float transmitDepthSky = CalculateCloudTransmitDepth(position, vec3(0.0, 1.0, 0.0), 8) * 0.2;
    #else
        float transmitDepthSky = CalculateCloudTransmitDepthSky(position);
    #endif

    #ifdef VC_GROUND_LIGHTING
        float transmitDepthGround = CalculateCloudTransmitDepth(position, vec3(0.0, -1.0, 0.0), 8);
    #else
        float transmitDepthGround = 0.0;
    #endif

    // Approximate inscattering probability
    float powderLight  = CalculatePowderEffect(transmitDepth);
    float powderView = CalculatePowderEffect(od);

    float powder = CalculatePowderEffect(powderLight, vDotL) *
                   CalculatePowderEffect(powderView, vDotL);

    #ifdef VC_MULTISCAT
        float an = 1.0, bn = 1.0;

        for (int i = 0; i < msSteps; ++i) {
            float phase = phases[i];

            CalculateCloudScattering(scatterCoeff, transmit, an, bn, transmitDepth, transmitDepthSky, transmitDepthGround, phase, powder, directScattering, skylightScattering, groundLightScattering, dlSteps);
        
            an *= cloudMultiScattCoeff_a;
            bn *= cloudMultiScattCoeff_b;
        }
    #else
        CalculateCloudScattering(scatterCoeff, transmit, 1.0, 1.0, transmitDepth, transmitDepthSky, transmitDepthGround, 1.0, powder, directScattering, skylightScattering, groundLightScattering, dlSteps);
    #endif
}

void calculateVolumetricClouds(inout vec3 cloud, inout float transmit, vec3 wDir, vec3 wLightVector, vec3 worldPosition, vec2 planetSphere, float dither, float vDotL, const float planetRadius, const int steps, const int dlSteps, const int alSteps){
    #ifndef VOLUMETRIC_CLOUDS
    	return;
    #endif

    // Early out when the clouds are behind the horizon or not visible.
    if ((eyeAltitude < volumetric_cloudMinHeight && planetSphere.y > 0.0)
     || (eyeAltitude > volumetric_cloudMaxHeight &&  wDir.y > 0.0)) return;

    // Calculate the cloud spheres.
    vec2 bottomSphere = rsi(vec3(0.0, planetRadius + eyeAltitude, 0.0), wDir, planetRadius + volumetric_cloudMinHeight);
    vec2    topSphere = rsi(vec3(0.0, planetRadius + eyeAltitude, 0.0), wDir, planetRadius + volumetric_cloudMaxHeight);

    // Get the distance from the eye to the start and endposition.
    float startDir = eyeAltitude > volumetric_cloudMaxHeight ? topSphere.x : bottomSphere.y;
    float   endDir = eyeAltitude > volumetric_cloudMaxHeight ? bottomSphere.x : topSphere.y;

    // Multiply the direction and distance by eachother to.
    vec3 startPos = wDir * startDir;
    vec3   endPos = wDir *   endDir;

    // Calculate the range of when the player is flying through the clouds.
    float marchRange = (1.0 - clamp01((eyeAltitude - volumetric_cloudMaxHeight) * 0.1)) * (1.0 - clamp01((volumetric_cloudMinHeight - eyeAltitude) * 0.1));

    const float inCloudMaxRayDistance = volumetric_cloudHeight * 10.0;

    float inCloudRaySpheres = (bottomSphere.y >= 0.0 && eyeAltitude > volumetric_cloudMinHeight ? bottomSphere.x : topSphere.y);
    float inCloudRayDistance = min(inCloudRaySpheres, inCloudMaxRayDistance);

    // Change the raymarcher's start and endposition when you fly through them or when geometry is behind it.
    startPos = startPos * (1.0 - marchRange);
      endPos = mix(endPos, inCloudRayDistance * wDir, marchRange);
	
    startPos += cameraPosition;
      endPos += cameraPosition;

    // Calculate the ray increment and the ray position.
    vec3  increment     = (endPos - startPos) / steps;
    vec3  cloudPosition = increment * dither + startPos;
    float rayLength     = length(increment);

    float   directScattering = 0.0;
    float skylightScattering = 0.0;
    float groundLightScattering = 0.0;

    float[VC_MULTISCAT_QUALITY] phases;

    // Calculate the cloud phase.
    #ifdef VC_MULTISCAT
        float phase = 1.0;
        CalculateMultipleScatteringCloudPhases(vDotL, phases);
    #else
        float phase = CalculateCloudPhase(vDotL);
    #endif

    vec3  cloudHitPos = vec3(0.0);
    float totTransmit = 0.0;

    float depthCompensation = sqrt(steps / (rayLength * 1.73205080757)); // 1.0 / sqrt(3) for alignment

    // Raymarch through the cloud volume & light the clouds contribution at each step
    for (int i = 0; i < steps; ++i, cloudPosition += increment){
        // Curve the cloud position around the Earth.
        vec3 curvedCloudPosition = cloudPosition;
             curvedCloudPosition.y = length(curvedCloudPosition + vec3(0.0, planetRadius, 0.0)) - planetRadius;

        float od = CalculateCloudOD(curvedCloudPosition, VC_NOISE_OCTAVES) * rayLength;
        if (od <= 0.0) { continue; } // Early out.
        
		float stepTransmit = exp2(-od * rLOG2);

        cloudHitPos += cloudPosition * transmit;
        totTransmit += transmit;

        CalculateCloudScattering(curvedCloudPosition, wLightVector, od * depthCompensation, vDotL, phases, transmit, stepTransmit, directScattering, skylightScattering, groundLightScattering, dlSteps, VC_MULTISCAT_QUALITY);

        transmit *= stepTransmit;
		
		if (transmit < 0.001) { transmit = 0.0; break; }
    }

    // Light the scattering and sum them up.
    vec3 directLighting =   directScattering * sunIlluminanceClouds;
    vec3 skyLighting    = skylightScattering * skyIlluminanceClouds;
    #ifdef VC_GROUND_LIGHTING
        vec3 groundLighting = groundLightScattering * sunIlluminanceClouds * abs(dot(vec3(0.0, 1.0, 0.0), wLightVector)) * rPI * 0.18;
    #else
        vec3 groundLighting = vec3(0.0);
    #endif
    
    #ifndef VC_MULTISCAT
        directLighting *= phase;
    #endif

    cloud = (skyLighting + groundLighting) * 0.25 * rPI + directLighting;

    cloudHitPos /= totTransmit;
    cloudHitPos -= cameraPosition;

	vec3 kCamera = vec3(0.0, 0.05 + cameraPosition.y * 0.001 + ATMOSPHERE.bottom_radius, 0.0);
	vec3 kPoint  = cloudHitPos * 0.001 + kCamera;
	
    vec3 transmitAP;
    vec3 in_scatter_moon;
    vec3 in_scatter = GetSkyRadianceToPoint(ATMOSPHERE, colortex7, colortex7, colortex7, kCamera, kPoint, 0.0, wSunVector, transmitAP, in_scatter_moon) * SKY_SPECTRAL_RADIANCE_TO_LUMINANCE;
    in_scatter_moon *= SKY_SPECTRAL_RADIANCE_TO_LUMINANCE_MOON;
    in_scatter += in_scatter_moon;
	
	if (any(isnan(in_scatter))) {
		in_scatter = vec3(0.0);
		transmitAP = vec3(1.0);
	}
	
	cloud = cloud * transmitAP + in_scatter * (1.0 - transmit);
    
    return;
}

// Absorb sunlight through the clouds.
float CalculateCloudShadows(vec3 position, vec3 direction, const int steps){
    #ifndef VC_SHADOWS
        return 1.0;
    #endif
    #ifndef VOLUMETRIC_CLOUDS
        return 1.0;
    #endif

    float rSteps = volumetric_cloudThickness / steps;
    float stepSize = rSteps / abs(direction.y);

    vec3 increment = direction * stepSize;

    float fade = smoothstep(0.125, 0.075, abs(direction.y));

    // Make sure the shadows keep on going even after we absorbed through the cloud.
    position += position.y <= volumetric_cloudMinHeight ? direction * (volumetric_cloudMinHeight - position.y) / direction.y : vec3(0.0);

    float transmit = 0.0;

    for (int i = 0; i < steps; ++i, position += increment){
        transmit += CalculateCloudOD(position, 3);
    }
	
    return exp2(-transmit * rLOG2 * stepSize) * (1.0 - fade) + fade;
}

#endif
