/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.ml.clustering;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.apache.commons.math3.exception.MathIllegalStateException;
import org.apache.commons.math3.exception.NumberIsTooSmallException;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.ml.clustering.CentroidCluster;
import org.apache.commons.math3.ml.clustering.Clusterable;
import org.apache.commons.math3.ml.clustering.Clusterer;
import org.apache.commons.math3.ml.clustering.DoublePoint;
import org.apache.commons.math3.ml.distance.DistanceMeasure;
import org.apache.commons.math3.ml.distance.EuclideanDistance;
import org.apache.commons.math3.random.JDKRandomGenerator;
import org.apache.commons.math3.random.RandomGenerator;
import org.apache.commons.math3.util.FastMath;
import org.apache.commons.math3.util.MathArrays;
import org.apache.commons.math3.util.MathUtils;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class FuzzyKMeansClusterer<T extends Clusterable>
extends Clusterer<T> {
    private static final double DEFAULT_EPSILON = 0.001;
    private final int k;
    private final int maxIterations;
    private final double fuzziness;
    private final double epsilon;
    private final RandomGenerator random;
    private double[][] membershipMatrix;
    private List<T> points;
    private List<CentroidCluster<T>> clusters;

    public FuzzyKMeansClusterer(int n, double d) {
        this(n, d, -1, new EuclideanDistance());
    }

    public FuzzyKMeansClusterer(int n, double d, int n2, DistanceMeasure distanceMeasure) {
        this(n, d, n2, distanceMeasure, 0.001, new JDKRandomGenerator());
    }

    public FuzzyKMeansClusterer(int n, double d, int n2, DistanceMeasure distanceMeasure, double d2, RandomGenerator randomGenerator) {
        super(distanceMeasure);
        if (d <= 1.0) {
            throw new NumberIsTooSmallException(d, (Number)1.0, false);
        }
        this.k = n;
        this.fuzziness = d;
        this.maxIterations = n2;
        this.epsilon = d2;
        this.random = randomGenerator;
        this.membershipMatrix = null;
        this.points = null;
        this.clusters = null;
    }

    public int getK() {
        return this.k;
    }

    public double getFuzziness() {
        return this.fuzziness;
    }

    public int getMaxIterations() {
        return this.maxIterations;
    }

    public double getEpsilon() {
        return this.epsilon;
    }

    public RandomGenerator getRandomGenerator() {
        return this.random;
    }

    public RealMatrix getMembershipMatrix() {
        if (this.membershipMatrix == null) {
            throw new MathIllegalStateException();
        }
        return MatrixUtils.createRealMatrix(this.membershipMatrix);
    }

    public List<T> getDataPoints() {
        return this.points;
    }

    public List<CentroidCluster<T>> getClusters() {
        return this.clusters;
    }

    public double getObjectiveFunctionValue() {
        if (this.points == null || this.clusters == null) {
            throw new MathIllegalStateException();
        }
        int n = 0;
        double d = 0.0;
        for (Clusterable clusterable : this.points) {
            int n2 = 0;
            for (CentroidCluster<T> centroidCluster : this.clusters) {
                double d2 = this.distance(clusterable, centroidCluster.getCenter());
                d += d2 * d2 * FastMath.pow(this.membershipMatrix[n][n2], this.fuzziness);
                ++n2;
            }
            ++n;
        }
        return d;
    }

    @Override
    public List<CentroidCluster<T>> cluster(Collection<T> collection) {
        int n;
        MathUtils.checkNotNull(collection);
        int n2 = collection.size();
        if (n2 < this.k) {
            throw new NumberIsTooSmallException(n2, (Number)this.k, false);
        }
        this.points = Collections.unmodifiableList(new ArrayList<T>(collection));
        this.clusters = new ArrayList<CentroidCluster<T>>();
        this.membershipMatrix = new double[n2][this.k];
        double[][] dArray = new double[n2][this.k];
        if (n2 == 0) {
            return this.clusters;
        }
        this.initializeMembershipMatrix();
        int n3 = ((Clusterable)this.points.get(0)).getPoint().length;
        for (n = 0; n < this.k; ++n) {
            this.clusters.add(new CentroidCluster(new DoublePoint(new double[n3])));
        }
        n = 0;
        int n4 = this.maxIterations < 0 ? Integer.MAX_VALUE : this.maxIterations;
        double d = 0.0;
        do {
            this.saveMembershipMatrix(dArray);
            this.updateClusterCenters();
            this.updateMembershipMatrix();
        } while ((d = this.calculateMaxMembershipChange(dArray)) > this.epsilon && ++n < n4);
        return this.clusters;
    }

    private void updateClusterCenters() {
        int n = 0;
        ArrayList<CentroidCluster<T>> arrayList = new ArrayList<CentroidCluster<T>>(this.k);
        for (CentroidCluster<T> centroidCluster : this.clusters) {
            Clusterable clusterable = centroidCluster.getCenter();
            int n2 = 0;
            double[] dArray = new double[clusterable.getPoint().length];
            double d = 0.0;
            for (Clusterable clusterable2 : this.points) {
                double d2 = FastMath.pow(this.membershipMatrix[n2][n], this.fuzziness);
                double[] dArray2 = clusterable2.getPoint();
                for (int i = 0; i < dArray.length; ++i) {
                    int n3 = i;
                    dArray[n3] = dArray[n3] + d2 * dArray2[i];
                }
                d += d2;
                ++n2;
            }
            MathArrays.scaleInPlace(1.0 / d, dArray);
            arrayList.add(new CentroidCluster(new DoublePoint(dArray)));
            ++n;
        }
        this.clusters.clear();
        this.clusters = arrayList;
    }

    private void updateMembershipMatrix() {
        for (int i = 0; i < this.points.size(); ++i) {
            Clusterable clusterable = (Clusterable)this.points.get(i);
            double d = Double.MIN_VALUE;
            int n = -1;
            for (int j = 0; j < this.clusters.size(); ++j) {
                double d2 = 0.0;
                double d3 = FastMath.abs(this.distance(clusterable, this.clusters.get(j).getCenter()));
                if (d3 != 0.0) {
                    for (CentroidCluster<T> centroidCluster : this.clusters) {
                        double d4 = FastMath.abs(this.distance(clusterable, centroidCluster.getCenter()));
                        if (d4 == 0.0) {
                            d2 = Double.POSITIVE_INFINITY;
                            break;
                        }
                        d2 += FastMath.pow(d3 / d4, 2.0 / (this.fuzziness - 1.0));
                    }
                }
                double d5 = d2 == 0.0 ? 1.0 : (d2 == Double.POSITIVE_INFINITY ? 0.0 : 1.0 / d2);
                this.membershipMatrix[i][j] = d5;
                if (!(this.membershipMatrix[i][j] > d)) continue;
                d = this.membershipMatrix[i][j];
                n = j;
            }
            this.clusters.get(n).addPoint(clusterable);
        }
    }

    private void initializeMembershipMatrix() {
        for (int i = 0; i < this.points.size(); ++i) {
            for (int j = 0; j < this.k; ++j) {
                this.membershipMatrix[i][j] = this.random.nextDouble();
            }
            this.membershipMatrix[i] = MathArrays.normalizeArray(this.membershipMatrix[i], 1.0);
        }
    }

    private double calculateMaxMembershipChange(double[][] dArray) {
        double d = 0.0;
        for (int i = 0; i < this.points.size(); ++i) {
            for (int j = 0; j < this.clusters.size(); ++j) {
                double d2 = FastMath.abs(this.membershipMatrix[i][j] - dArray[i][j]);
                d = FastMath.max(d2, d);
            }
        }
        return d;
    }

    private void saveMembershipMatrix(double[][] dArray) {
        for (int i = 0; i < this.points.size(); ++i) {
            System.arraycopy(this.membershipMatrix[i], 0, dArray[i], 0, this.clusters.size());
        }
    }
}

