/*
 * Decompiled with CFR 0.152.
 */
package org.kingdoms.managers.backup;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import org.kingdoms.locale.MessageHandler;
import org.kingdoms.utils.time.TimeUtils;

public abstract class BackupManager {
    protected static final DateTimeFormatter DATE_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final int METHOD = 9;
    private static final String EXTENSION = ".zip";
    protected static boolean useMultiBackups = true;
    protected final Path backups;
    protected final Path toBackup;
    private final Executor executor;

    public BackupManager(Path backups, Path toBackup, Executor executor) {
        this.toBackup = Objects.requireNonNull(toBackup, "Cannot backup null directory");
        this.backups = Objects.requireNonNull(backups, "Backups directory cannot be null");
        this.executor = executor;
        this.validateDir();
    }

    private void validateDir() {
        try {
            Files.createDirectories(this.backups, new FileAttribute[0]);
        }
        catch (IOException e) {
            MessageHandler.sendConsolePluginMessage("&4Failed to create backups directory.");
            e.printStackTrace();
        }
    }

    public String getDate() {
        return LocalDateTime.now().format(DATE_PATTERN);
    }

    public CompletableFuture<Void> unzipFiles(Path zip, Path unzipTo) {
        Objects.requireNonNull(zip, "Cannot unzip null directory.");
        Objects.requireNonNull(unzipTo, "Cannot unzip to null directory");
        if (!zip.toString().toLowerCase().endsWith(EXTENSION)) {
            throw new IllegalArgumentException("ZIP path must refer to a ZIP file");
        }
        if (Files.exists(unzipTo, new LinkOption[0]) && !Files.isDirectory(unzipTo, new LinkOption[0])) {
            throw new IllegalArgumentException("Cannot unzip to a non-directory");
        }
        return CompletableFuture.runAsync(() -> {
            try {
                Files.createDirectories(unzipTo, new FileAttribute[0]);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            try (ZipFile zipFile = new ZipFile(zip.toString());){
                Enumeration<? extends ZipEntry> entries = zipFile.entries();
                while (entries.hasMoreElements()) {
                    ZipEntry entry = entries.nextElement();
                    InputStream stream = zipFile.getInputStream(entry);
                    try {
                        Path path = unzipTo.resolve(entry.getName());
                        Files.createDirectories(path, new FileAttribute[0]);
                        Files.copy(stream, path, StandardCopyOption.REPLACE_EXISTING);
                    }
                    finally {
                        if (stream == null) continue;
                        stream.close();
                    }
                }
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }, this.executor);
    }

    public void takeBackup() {
        if (!useMultiBackups && this.hasBackupToday()) {
            return;
        }
        this.deleteOldBackups(30, TimeUnit.DAYS);
        this.zipFiles();
    }

    public CompletableFuture<Integer> zipFiles() {
        this.validateDir();
        Path sourcePath = this.getZip();
        try {
            Files.createFile(sourcePath, new FileAttribute[0]);
        }
        catch (IOException ex) {
            MessageHandler.sendConsolePluginMessage("&4Error while attempting to create ZIP file.");
            ex.printStackTrace();
        }
        return CompletableFuture.supplyAsync(() -> {
            final AtomicInteger backedUp = new AtomicInteger();
            try (final ZipOutputStream zs = new ZipOutputStream(new BufferedOutputStream(Files.newOutputStream(sourcePath, new OpenOption[0])));){
                zs.setLevel(9);
                zs.setComment("A backup file for Kingdoms minecraft plugin data.\nThese backups contain language file configs and config.yml, kingdoms, players and lands data\ndepending on the options specified in the config.\n\nNote that you have to stop the server before restoring one of these backups.\nBackup taken at: " + TimeUtils.getDateAndTime());
                Files.walkFileTree(this.toBackup, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                        return BackupManager.this.isWhitelistedDirectory(dir) || dir == BackupManager.this.toBackup ? FileVisitResult.CONTINUE : FileVisitResult.SKIP_SUBTREE;
                    }

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                        if (!BackupManager.this.isWhitelistedFile(file)) {
                            return FileVisitResult.SKIP_SUBTREE;
                        }
                        ZipEntry zipEntry = new ZipEntry(BackupManager.this.toBackup.relativize(file).toString());
                        try {
                            zs.putNextEntry(zipEntry);
                            Files.copy(file, zs);
                            zs.closeEntry();
                            backedUp.getAndIncrement();
                        }
                        catch (IOException e) {
                            MessageHandler.sendConsolePluginMessage("&4Error while attempting to backup a file&8: &e" + file.getFileName());
                            e.printStackTrace();
                        }
                        return FileVisitResult.CONTINUE;
                    }
                });
                Collection<Path> additionalStreams = this.addToZip();
                if (additionalStreams != null) {
                    for (Path additional : additionalStreams) {
                        ZipEntry zipEntry = new ZipEntry(this.toBackup.relativize(additional).toString());
                        try {
                            zs.putNextEntry(zipEntry);
                            Files.copy(additional, zs);
                            zs.closeEntry();
                            backedUp.getAndIncrement();
                        }
                        catch (IOException e) {
                            MessageHandler.sendConsolePluginMessage("&4Error while attempting to backup a file&8: &e" + additional.getFileName());
                            e.printStackTrace();
                        }
                    }
                }
            }
            catch (IOException e) {
                MessageHandler.sendConsolePluginMessage("&4Error while attempting to take a backup...");
                e.printStackTrace();
            }
            return backedUp.get();
        }, this.executor);
    }

    public Path getMultiZipName() {
        Path file = this.getZipPath();
        Path parent = file.getParent();
        String name = file.getFileName().toString();
        name = name.substring(0, name.lastIndexOf(46));
        String start = name + " (";
        String end = ").zip";
        int count = 1;
        while (Files.exists(file, new LinkOption[0])) {
            file = parent.resolve(start + count++ + end);
        }
        return file;
    }

    public abstract boolean isWhitelistedDirectory(Path var1);

    public abstract boolean isWhitelistedFile(Path var1);

    public abstract Collection<Path> addToZip();

    public boolean shouldBeDeleted(Path path, int time, TimeUnit timeUnit) {
        long created;
        try {
            created = Files.getLastModifiedTime(path, new LinkOption[0]).to(TimeUnit.MILLISECONDS);
        }
        catch (IOException e) {
            e.printStackTrace();
            return false;
        }
        long diff = System.currentTimeMillis() - created;
        return diff >= timeUnit.toMillis(time);
    }

    public CompletableFuture<Void> deleteOldBackups(int time, TimeUnit timeUnit) {
        if (time <= 0) {
            return null;
        }
        return CompletableFuture.runAsync(() -> {
            try {
                Files.walk(this.backups, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(f -> {
                    if (this.shouldBeDeleted((Path)f, time, timeUnit)) {
                        try {
                            MessageHandler.sendConsolePluginMessage("&2Deleting old backup... &6" + f.getFileName());
                            Files.delete(f);
                        }
                        catch (IOException e) {
                            e.printStackTrace();
                        }
                    }
                });
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        });
    }

    public Path getZipPath() {
        return this.backups.resolve(this.getDate() + EXTENSION);
    }

    public Path getZip() {
        return useMultiBackups ? this.getMultiZipName() : this.getZipPath();
    }

    public boolean hasBackupToday() {
        return Files.exists(this.getZip(), new LinkOption[0]);
    }
}

