/*
 * Decompiled with CFR 0.152.
 */
package net.skinsrestorer.shared.storage.adapter.file;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.UserDefinedFileAttributeView;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.skinsrestorer.api.property.SkinProperty;
import net.skinsrestorer.api.property.SkinType;
import net.skinsrestorer.api.property.SkinVariant;
import net.skinsrestorer.shadow.configme.SettingsManager;
import net.skinsrestorer.shadow.google.gson.Gson;
import net.skinsrestorer.shadow.google.gson.GsonBuilder;
import net.skinsrestorer.shadow.javax.inject.Inject;
import net.skinsrestorer.shared.config.GUIConfig;
import net.skinsrestorer.shared.log.SRLogger;
import net.skinsrestorer.shared.plugin.SRPlugin;
import net.skinsrestorer.shared.storage.adapter.StorageAdapter;
import net.skinsrestorer.shared.storage.adapter.file.model.cache.MojangCacheFile;
import net.skinsrestorer.shared.storage.adapter.file.model.player.LegacyPlayerFile;
import net.skinsrestorer.shared.storage.adapter.file.model.player.PlayerFile;
import net.skinsrestorer.shared.storage.adapter.file.model.skin.CustomSkinFile;
import net.skinsrestorer.shared.storage.adapter.file.model.skin.LegacySkinFile;
import net.skinsrestorer.shared.storage.adapter.file.model.skin.PlayerSkinFile;
import net.skinsrestorer.shared.storage.adapter.file.model.skin.URLIndexFile;
import net.skinsrestorer.shared.storage.adapter.file.model.skin.URLSkinFile;
import net.skinsrestorer.shared.storage.model.cache.MojangCacheData;
import net.skinsrestorer.shared.storage.model.player.LegacyPlayerData;
import net.skinsrestorer.shared.storage.model.player.PlayerData;
import net.skinsrestorer.shared.storage.model.skin.CustomSkinData;
import net.skinsrestorer.shared.storage.model.skin.LegacySkinData;
import net.skinsrestorer.shared.storage.model.skin.PlayerSkinData;
import net.skinsrestorer.shared.storage.model.skin.URLIndexData;
import net.skinsrestorer.shared.storage.model.skin.URLSkinData;
import net.skinsrestorer.shared.utils.SRFileUtils;

public class FileAdapter
implements StorageAdapter {
    private static final String LAST_KNOW_NAME_ATTRIBUTE = "sr_last_known_name";
    private static final Pattern UUID_REGEX = Pattern.compile("^[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{12}$");
    private final Path skinsFolder;
    private final Path playersFolder;
    private final Path cacheFolder;
    private final Path legacyFolder;
    private final SettingsManager settings;
    private final Gson gson = new GsonBuilder().disableHtmlEscaping().create();
    private final SRLogger logger;

    @Inject
    public FileAdapter(SRPlugin plugin, SettingsManager settings, SRLogger logger) {
        Path dataFolder = plugin.getDataFolder();
        this.skinsFolder = dataFolder.resolve("skins");
        this.playersFolder = dataFolder.resolve("players");
        this.cacheFolder = dataFolder.resolve("cache");
        this.legacyFolder = dataFolder.resolve("legacy");
        this.settings = settings;
        this.logger = logger;
        try {
            this.migrate(dataFolder);
        }
        catch (IOException e) {
            throw new IllegalStateException(e);
        }
        this.init();
    }

    private static String hashSHA256(String input) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hash = digest.digest(input.getBytes(StandardCharsets.UTF_8));
            return FileAdapter.bytesToHex(hash);
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    private static String bytesToHex(byte[] hash) {
        StringBuilder hexString = new StringBuilder(2 * hash.length);
        for (byte b : hash) {
            String hex = Integer.toHexString(0xFF & b);
            if (hex.length() == 1) {
                hexString.append('0');
            }
            hexString.append(hex);
        }
        return hexString.toString();
    }

    @Override
    public void init() {
        try {
            Files.createDirectories(this.skinsFolder, new FileAttribute[0]);
            Files.createDirectories(this.playersFolder, new FileAttribute[0]);
            Files.createDirectories(this.cacheFolder, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void migrate(Path dataFolder) throws IOException {
        SRFileUtils.renameFile(dataFolder, "Skins", "skins");
        SRFileUtils.renameFile(dataFolder, "Players", "players");
        this.migrateSkins();
        this.migratePlayers();
    }

    private void migratePlayers() {
        if (!Files.exists(this.playersFolder, new LinkOption[0])) {
            return;
        }
        Path legacyPlayersFolder = this.legacyFolder.resolve("players");
        boolean generatedFolder = false;
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.playersFolder, "*.player");){
            for (Path path : stream) {
                String fileName = path.getFileName().toString();
                String playerName = fileName.substring(0, fileName.length() - ".player".length());
                if (UUID_REGEX.matcher(playerName).matches()) {
                    return;
                }
                if (!generatedFolder) {
                    generatedFolder = true;
                    Files.createDirectories(legacyPlayersFolder, new FileAttribute[0]);
                    this.logger.info("Migrating legacy player files to new format...");
                }
                try {
                    Path legacyPlayerFile = this.resolveLegacyPlayerFile(playerName);
                    if (Files.exists(legacyPlayerFile, new LinkOption[0])) continue;
                    String skinName = new String(Files.readAllBytes(path), StandardCharsets.UTF_8);
                    LegacyPlayerData legacyPlayerData = LegacyPlayerData.of(playerName, skinName);
                    Files.write(legacyPlayerFile, this.gson.toJson(LegacyPlayerFile.fromLegacyPlayerData(legacyPlayerData)).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
                    Files.deleteIfExists(path);
                }
                catch (Exception e) {
                    this.logger.warning("Failed to migrate legacy player file: " + path.getFileName(), e);
                }
            }
        }
        catch (IOException e) {
            this.logger.warning("Failed to migrate legacy player files", e);
        }
        if (generatedFolder) {
            this.logger.info("Player files migration complete!");
        }
    }

    private void migrateSkins() {
        if (!Files.exists(this.skinsFolder, new LinkOption[0])) {
            return;
        }
        Path legacySkinsFolder = this.legacyFolder.resolve("skins");
        boolean generatedFolder = false;
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.skinsFolder, "*.skin");){
            for (Path path : stream) {
                if (!generatedFolder) {
                    generatedFolder = true;
                    Files.createDirectories(legacySkinsFolder, new FileAttribute[0]);
                    this.logger.info("Migrating legacy skin files to new format...");
                }
                try {
                    String fileName = path.getFileName().toString();
                    String skinName = fileName.substring(0, fileName.length() - ".skin".length());
                    Path legacySkinFile = this.resolveLegacySkinFile(skinName);
                    if (Files.exists(legacySkinFile, new LinkOption[0])) continue;
                    String[] lines = new String(Files.readAllBytes(path), StandardCharsets.UTF_8).split("\n");
                    String skinValue = lines[0].trim();
                    String skinSignature = lines[1].trim();
                    SkinProperty skinProperty = SkinProperty.of(skinValue, skinSignature);
                    if (lines.length == 2 || this.isLegacyCustomSkinTimestamp(Long.parseLong(lines[2].trim()))) {
                        this.setCustomSkinData(skinName, CustomSkinData.of(skinName, skinProperty));
                    } else {
                        LegacySkinData legacySkinData = LegacySkinData.of(skinName, skinProperty);
                        Files.write(legacySkinFile, this.gson.toJson(LegacySkinFile.fromLegacySkinData(legacySkinData)).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
                    }
                    Files.deleteIfExists(path);
                }
                catch (Exception e) {
                    this.logger.warning("Failed to migrate legacy skin file: " + path.getFileName(), e);
                }
            }
        }
        catch (IOException e) {
            this.logger.warning("Failed to migrate legacy skin files", e);
        }
        if (generatedFolder) {
            this.logger.info("Skin files migration complete!");
        }
    }

    @Override
    public Optional<PlayerData> getPlayerData(UUID uuid) throws StorageAdapter.StorageException {
        Path playerFile = this.resolvePlayerFile(uuid);
        if (!Files.exists(playerFile, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String json = new String(Files.readAllBytes(playerFile), StandardCharsets.UTF_8);
            PlayerFile file = this.gson.fromJson(json, PlayerFile.class);
            return Optional.of(file.toPlayerData());
        }
        catch (Exception e) {
            throw new StorageAdapter.StorageException(e);
        }
    }

    @Override
    public void setPlayerData(UUID uuid, PlayerData data) {
        Path playerFile = this.resolvePlayerFile(uuid);
        try {
            PlayerFile file = PlayerFile.fromPlayerData(data);
            Files.write(playerFile, this.gson.toJson(file).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            this.logger.warning("Failed to save player data for " + uuid, e);
        }
    }

    @Override
    public Optional<PlayerSkinData> getPlayerSkinData(UUID uuid) throws StorageAdapter.StorageException {
        Path skinFile = this.resolvePlayerSkinFile(uuid);
        if (!Files.exists(skinFile, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String json = new String(Files.readAllBytes(skinFile), StandardCharsets.UTF_8);
            PlayerSkinFile file = this.gson.fromJson(json, PlayerSkinFile.class);
            return Optional.of(file.toPlayerSkinData());
        }
        catch (Exception e) {
            throw new StorageAdapter.StorageException(e);
        }
    }

    @Override
    public void removePlayerSkinData(UUID uuid) {
        Path skinFile = this.resolvePlayerSkinFile(uuid);
        try {
            Files.deleteIfExists(skinFile);
        }
        catch (IOException e) {
            this.logger.warning("Failed to remove player skin data for " + uuid, e);
        }
    }

    @Override
    public void setPlayerSkinData(UUID uuid, PlayerSkinData skinData) {
        Path skinFile = this.resolvePlayerSkinFile(uuid);
        try {
            PlayerSkinFile file = PlayerSkinFile.fromPlayerSkinData(skinData);
            Files.write(skinFile, this.gson.toJson(file).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            UserDefinedFileAttributeView view = Files.getFileAttributeView(skinFile, UserDefinedFileAttributeView.class, new LinkOption[0]);
            view.write(LAST_KNOW_NAME_ATTRIBUTE, StandardCharsets.UTF_8.encode(skinData.getLastKnownName()));
        }
        catch (IOException e) {
            this.logger.warning("Failed to save player skin data for " + uuid, e);
        }
    }

    @Override
    public Optional<URLSkinData> getURLSkinData(String url, SkinVariant skinVariant) throws StorageAdapter.StorageException {
        Path skinFile = this.resolveURLSkinFile(url, skinVariant);
        if (!Files.exists(skinFile, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String json = new String(Files.readAllBytes(skinFile), StandardCharsets.UTF_8);
            URLSkinFile file = this.gson.fromJson(json, URLSkinFile.class);
            return Optional.of(file.toURLSkinData());
        }
        catch (Exception e) {
            throw new StorageAdapter.StorageException(e);
        }
    }

    @Override
    public void removeURLSkinData(String url, SkinVariant skinVariant) {
        Path skinFile = this.resolveURLSkinFile(url, skinVariant);
        try {
            Files.deleteIfExists(skinFile);
        }
        catch (IOException e) {
            this.logger.warning("Failed to remove URL skin data for " + url, e);
        }
    }

    @Override
    public void setURLSkinData(String url, URLSkinData skinData) {
        Path skinFile = this.resolveURLSkinFile(url, skinData.getSkinVariant());
        try {
            URLSkinFile file = URLSkinFile.fromURLSkinData(skinData);
            Files.write(skinFile, this.gson.toJson(file).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            this.logger.warning("Failed to save URL skin data for " + url, e);
        }
    }

    @Override
    public Optional<URLIndexData> getURLSkinIndex(String url) throws StorageAdapter.StorageException {
        Path skinFile = this.resolveURLSkinIndexFile(url);
        if (!Files.exists(skinFile, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String json = new String(Files.readAllBytes(skinFile), StandardCharsets.UTF_8);
            URLIndexFile file = this.gson.fromJson(json, URLIndexFile.class);
            return Optional.of(file.toURLIndexData());
        }
        catch (Exception e) {
            throw new StorageAdapter.StorageException(e);
        }
    }

    @Override
    public void removeURLSkinIndex(String url) {
        Path skinFile = this.resolveURLSkinIndexFile(url);
        try {
            Files.deleteIfExists(skinFile);
        }
        catch (IOException e) {
            this.logger.warning("Failed to remove URL skin index for " + url, e);
        }
    }

    @Override
    public void setURLSkinIndex(String url, URLIndexData skinData) {
        Path skinFile = this.resolveURLSkinIndexFile(url);
        try {
            URLIndexFile file = URLIndexFile.fromURLIndexData(skinData);
            Files.write(skinFile, this.gson.toJson(file).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            this.logger.warning("Failed to save URL skin index for " + url, e);
        }
    }

    @Override
    public Optional<CustomSkinData> getCustomSkinData(String skinName) throws StorageAdapter.StorageException {
        Path skinFile = this.resolveCustomSkinFile(skinName = CustomSkinData.sanitizeCustomSkinName(skinName));
        if (!Files.exists(skinFile, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String json = new String(Files.readAllBytes(skinFile), StandardCharsets.UTF_8);
            CustomSkinFile file = this.gson.fromJson(json, CustomSkinFile.class);
            return Optional.of(file.toCustomSkinData());
        }
        catch (Exception e) {
            throw new StorageAdapter.StorageException(e);
        }
    }

    @Override
    public void removeCustomSkinData(String skinName) {
        skinName = CustomSkinData.sanitizeCustomSkinName(skinName);
        Path skinFile = this.resolveCustomSkinFile(skinName);
        try {
            Files.deleteIfExists(skinFile);
        }
        catch (IOException e) {
            this.logger.warning("Failed to remove custom skin data for " + skinName, e);
        }
    }

    @Override
    public void setCustomSkinData(String skinName, CustomSkinData skinData) {
        skinName = CustomSkinData.sanitizeCustomSkinName(skinName);
        Path skinFile = this.resolveCustomSkinFile(skinName);
        try {
            CustomSkinFile file = CustomSkinFile.fromCustomSkinData(skinData);
            Files.write(skinFile, this.gson.toJson(file).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            this.logger.warning("Failed to save custom skin data for " + skinName, e);
        }
    }

    @Override
    public Optional<LegacySkinData> getLegacySkinData(String skinName) throws StorageAdapter.StorageException {
        Path skinFile = this.resolveLegacySkinFile(skinName = this.sanitizeLegacySkinName(skinName));
        if (!Files.exists(skinFile, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String json = new String(Files.readAllBytes(skinFile), StandardCharsets.UTF_8);
            LegacySkinFile file = this.gson.fromJson(json, LegacySkinFile.class);
            return Optional.of(file.toLegacySkinData());
        }
        catch (Exception e) {
            throw new StorageAdapter.StorageException(e);
        }
    }

    @Override
    public void removeLegacySkinData(String skinName) {
        skinName = this.sanitizeLegacySkinName(skinName);
        Path skinFile = this.resolveLegacySkinFile(skinName);
        try {
            Files.deleteIfExists(skinFile);
        }
        catch (IOException e) {
            this.logger.warning("Failed to remove legacy skin data for " + skinName, e);
        }
    }

    @Override
    public Optional<LegacyPlayerData> getLegacyPlayerData(String playerName) throws StorageAdapter.StorageException {
        Path legacyFile = this.resolveLegacyPlayerFile(playerName = this.sanitizeLegacyPlayerName(playerName));
        if (!Files.exists(legacyFile, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String json = new String(Files.readAllBytes(legacyFile), StandardCharsets.UTF_8);
            LegacyPlayerFile file = this.gson.fromJson(json, LegacyPlayerFile.class);
            return Optional.of(file.toLegacyPlayerData());
        }
        catch (Exception e) {
            throw new StorageAdapter.StorageException(e);
        }
    }

    @Override
    public void removeLegacyPlayerData(String playerName) {
        playerName = this.sanitizeLegacyPlayerName(playerName);
        Path legacyFile = this.resolveLegacyPlayerFile(playerName);
        try {
            Files.deleteIfExists(legacyFile);
        }
        catch (IOException e) {
            this.logger.warning("Failed to remove legacy player data for " + playerName, e);
        }
    }

    @Override
    public Map<String, String> getStoredGUISkins(int offset) {
        LinkedHashMap<String, String> list = new LinkedHashMap<String, String>();
        Map<String, GUIFileData> files = this.getGUIFilesSorted(offset);
        for (Map.Entry<String, GUIFileData> entry : files.entrySet()) {
            if (entry.getValue().getSkinType() == SkinType.PLAYER) {
                try {
                    this.getPlayerSkinData(UUID.fromString(entry.getValue().getFileName())).ifPresent(skinData -> list.put((String)entry.getKey(), skinData.getProperty().getValue()));
                }
                catch (StorageAdapter.StorageException e) {
                    this.logger.warning("Failed to load player skin data for " + entry.getValue().getFileName(), e);
                }
                continue;
            }
            if (entry.getValue().getSkinType() != SkinType.CUSTOM) continue;
            try {
                this.getCustomSkinData(entry.getValue().getFileName()).ifPresent(skinData -> list.put((String)entry.getKey(), skinData.getProperty().getValue()));
            }
            catch (StorageAdapter.StorageException e) {
                this.logger.warning("Failed to load custom skin data for " + entry.getValue().getFileName(), e);
            }
        }
        return list;
    }

    private Map<String, GUIFileData> getGUIFilesSorted(int offset) {
        boolean customEnabled = this.settings.getProperty(GUIConfig.CUSTOM_GUI_ENABLED);
        boolean customOnly = this.settings.getProperty(GUIConfig.CUSTOM_GUI_ONLY);
        List customSkins = this.settings.getProperty(GUIConfig.CUSTOM_GUI_SKINS).stream().map(s -> s.toLowerCase(Locale.ROOT)).distinct().collect(Collectors.toList());
        TreeMap<String, GUIFileData> files = new TreeMap<String, GUIFileData>(String.CASE_INSENSITIVE_ORDER);
        try (Stream<Path> stream = Files.walk(this.skinsFolder, 1, new FileVisitOption[0]);){
            int skinIndex = 0;
            Iterator it = stream.iterator();
            while (it.hasNext()) {
                String fileName;
                int lastDotIndex;
                Path path = (Path)it.next();
                if (Files.isDirectory(path, new LinkOption[0]) || (lastDotIndex = (fileName = path.getFileName().toString()).lastIndexOf(".")) == -1) continue;
                String extension = fileName.substring(lastDotIndex + 1);
                String name = fileName.substring(0, lastDotIndex);
                boolean isPlayerSkin = extension.equals("playerskin");
                boolean isCustomSkin = extension.equals("customskin");
                if (!isPlayerSkin && !isCustomSkin || isPlayerSkin && customEnabled && customOnly || isCustomSkin && !customEnabled || customEnabled && customOnly && !customSkins.contains(name.toLowerCase(Locale.ROOT))) continue;
                if (skinIndex < offset) {
                    ++skinIndex;
                    continue;
                }
                GUIFileData data = new GUIFileData(name, path, isPlayerSkin ? SkinType.PLAYER : SkinType.CUSTOM);
                if (isPlayerSkin) {
                    this.getLastKnownName(path).ifPresent(lastKnownName -> files.put((String)lastKnownName, data));
                } else {
                    files.put(name, data);
                }
                if (skinIndex++ < offset + 36) continue;
                break;
            }
        }
        catch (IOException e) {
            this.logger.warning("Failed to load GUI files", e);
        }
        return files;
    }

    private Optional<String> getLastKnownName(Path path) {
        try {
            UserDefinedFileAttributeView view = Files.getFileAttributeView(path, UserDefinedFileAttributeView.class, new LinkOption[0]);
            if (!view.list().contains(LAST_KNOW_NAME_ATTRIBUTE)) {
                return Optional.empty();
            }
            ByteBuffer buffer = ByteBuffer.allocate(view.size(LAST_KNOW_NAME_ATTRIBUTE));
            view.read(LAST_KNOW_NAME_ATTRIBUTE, buffer);
            buffer.flip();
            return Optional.of(StandardCharsets.UTF_8.decode(buffer).toString());
        }
        catch (IOException e) {
            this.logger.warning("Failed to load last known name for " + path.getFileName(), e);
            return Optional.empty();
        }
    }

    @Override
    public void purgeStoredOldSkins(long targetPurgeTimestamp) throws StorageAdapter.StorageException {
        ArrayList files = new ArrayList();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.skinsFolder, "*.playerskin");){
            stream.forEach(files::add);
        }
        catch (IOException e) {
            throw new StorageAdapter.StorageException(e);
        }
        for (Path file : files) {
            try {
                String json = new String(Files.readAllBytes(file), StandardCharsets.UTF_8);
                PlayerSkinFile skinFile = this.gson.fromJson(json, PlayerSkinFile.class);
                if (skinFile.getTimestamp() == 0L || skinFile.getTimestamp() >= targetPurgeTimestamp) continue;
                Files.deleteIfExists(file);
            }
            catch (Exception e) {
                throw new StorageAdapter.StorageException(e);
            }
        }
    }

    @Override
    public Optional<MojangCacheData> getCachedUUID(String playerName) throws StorageAdapter.StorageException {
        Path cacheFile = this.resolveCacheFile(playerName);
        if (!Files.exists(cacheFile, new LinkOption[0])) {
            return Optional.empty();
        }
        try {
            String json = new String(Files.readAllBytes(cacheFile), StandardCharsets.UTF_8);
            MojangCacheFile file = this.gson.fromJson(json, MojangCacheFile.class);
            return Optional.of(file.toCacheData());
        }
        catch (Exception e) {
            throw new StorageAdapter.StorageException(e);
        }
    }

    @Override
    public void setCachedUUID(String playerName, MojangCacheData mojangCacheData) {
        Path cacheFile = this.resolveCacheFile(playerName);
        try {
            MojangCacheFile file = MojangCacheFile.fromMojangCacheData(mojangCacheData);
            Files.write(cacheFile, this.gson.toJson(file).getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        catch (IOException e) {
            this.logger.warning("Failed to save cached UUID for " + playerName, e);
        }
    }

    private Path resolveCustomSkinFile(String skinName) {
        return this.skinsFolder.resolve(skinName + ".customskin");
    }

    private Path resolveLegacySkinFile(String skinName) {
        return this.legacyFolder.resolve("skins").resolve(skinName + ".legacyskin");
    }

    private Path resolveURLSkinFile(String url, SkinVariant skinVariant) {
        return this.skinsFolder.resolve(FileAdapter.hashSHA256(url) + "_" + skinVariant.name() + ".urlskin");
    }

    private Path resolveURLSkinIndexFile(String url) {
        return this.skinsFolder.resolve(FileAdapter.hashSHA256(url) + ".urlindex");
    }

    private Path resolvePlayerSkinFile(UUID uuid) {
        return this.skinsFolder.resolve(uuid + ".playerskin");
    }

    private Path resolvePlayerFile(UUID uuid) {
        return this.playersFolder.resolve(uuid + ".player");
    }

    private Path resolveLegacyPlayerFile(String name) {
        return this.legacyFolder.resolve("players").resolve(name + ".legacyplayer");
    }

    private Path resolveCacheFile(String name) {
        return this.cacheFolder.resolve(name + ".mojangcache");
    }

    private String sanitizeLegacyPlayerName(String playerName) {
        return playerName.toLowerCase();
    }

    private String sanitizeLegacySkinName(String skinName) {
        return skinName.toLowerCase();
    }

    private static class GUIFileData {
        private final String fileName;
        private final Path path;
        private final SkinType skinType;

        public String getFileName() {
            return this.fileName;
        }

        public Path getPath() {
            return this.path;
        }

        public SkinType getSkinType() {
            return this.skinType;
        }

        public GUIFileData(String fileName, Path path, SkinType skinType) {
            this.fileName = fileName;
            this.path = path;
            this.skinType = skinType;
        }
    }
}

