/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.data;

import com.google.common.collect.ImmutableMap;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.mojang.logging.LogUtils;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nullable;
import net.minecraft.WorldVersion;
import net.minecraft.data.CachedOutput;
import org.apache.commons.lang3.mutable.MutableInt;
import org.slf4j.Logger;

public class HashCache {
    static final Logger LOGGER = LogUtils.getLogger();
    private static final String HEADER_MARKER = "// ";
    private final Path rootDir;
    private final Path cacheDir;
    private final String versionId;
    private final Map<String, ProviderCache> caches;
    private final Set<String> cachesToWrite = new HashSet<String>();
    final Set<Path> cachePaths = new HashSet<Path>();
    private final int initialCount;
    private int writes;

    private Path getProviderCachePath(String providerName) {
        return this.cacheDir.resolve(Hashing.sha1().hashString((CharSequence)providerName, StandardCharsets.UTF_8).toString());
    }

    public HashCache(Path root, Collection<String> providerNames, WorldVersion gameVersion) throws IOException {
        this.versionId = gameVersion.getName();
        this.rootDir = root;
        this.cacheDir = root.resolve(".cache");
        Files.createDirectories(this.cacheDir, new FileAttribute[0]);
        HashMap<String, ProviderCache> map = new HashMap<String, ProviderCache>();
        int i = 0;
        for (String string : providerNames) {
            Path path = this.getProviderCachePath(string);
            this.cachePaths.add(path);
            ProviderCache providerCache = HashCache.readCache(root, path);
            map.put(string, providerCache);
            i += providerCache.count();
        }
        this.caches = map;
        this.initialCount = i;
    }

    private static ProviderCache readCache(Path root, Path dataProviderPath) {
        if (Files.isReadable(dataProviderPath)) {
            try {
                return ProviderCache.load(root, dataProviderPath);
            }
            catch (Exception exception) {
                LOGGER.warn("Failed to parse cache {}, discarding", (Object)dataProviderPath, (Object)exception);
            }
        }
        return new ProviderCache("unknown", (ImmutableMap<Path, HashCode>)ImmutableMap.of());
    }

    public boolean shouldRunInThisVersion(String providerName) {
        ProviderCache providerCache = this.caches.get(providerName);
        return providerCache == null || !providerCache.version.equals(this.versionId);
    }

    public CompletableFuture<UpdateResult> generateUpdate(String providerName, UpdateFunction runner) {
        ProviderCache providerCache = this.caches.get(providerName);
        if (providerCache == null) {
            throw new IllegalStateException("Provider not registered: " + providerName);
        }
        CacheUpdater cacheUpdater = new CacheUpdater(this, providerName, this.versionId, providerCache);
        return runner.update(cacheUpdater).thenApply(void_ -> cacheUpdater.close());
    }

    public void applyUpdate(UpdateResult runResult) {
        this.caches.put(runResult.providerId(), runResult.cache());
        this.cachesToWrite.add(runResult.providerId());
        this.writes += runResult.writes();
    }

    public void purgeStaleAndWrite() throws IOException {
        final HashSet<Path> set = new HashSet<Path>();
        this.caches.forEach((providerName, cachedData) -> {
            if (this.cachesToWrite.contains(providerName)) {
                Path path = this.getProviderCachePath((String)providerName);
                cachedData.save(this.rootDir, path, DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(LocalDateTime.now()) + "\t" + providerName);
            }
            set.addAll((Collection<Path>)cachedData.data().keySet());
        });
        set.add(this.rootDir.resolve("version.json"));
        final MutableInt mutableInt = new MutableInt();
        final MutableInt mutableInt2 = new MutableInt();
        Files.walkFileTree(this.rootDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            @Override
            public FileVisitResult visitFile(Path path, BasicFileAttributes basicFileAttributes) {
                if (HashCache.this.cachePaths.contains(path)) {
                    return FileVisitResult.CONTINUE;
                }
                mutableInt.increment();
                if (set.contains(path)) {
                    return FileVisitResult.CONTINUE;
                }
                try {
                    Files.delete(path);
                }
                catch (IOException iOException) {
                    LOGGER.warn("Failed to delete file {}", (Object)path, (Object)iOException);
                }
                mutableInt2.increment();
                return FileVisitResult.CONTINUE;
            }

            @Override
            public /* synthetic */ FileVisitResult visitFile(Object object, BasicFileAttributes basicFileAttributes) throws IOException {
                return this.visitFile((Path)object, basicFileAttributes);
            }
        });
        LOGGER.info("Caching: total files: {}, old count: {}, new count: {}, removed stale: {}, written: {}", new Object[]{mutableInt, this.initialCount, set.size(), mutableInt2, this.writes});
    }

    record ProviderCache(String version, ImmutableMap<Path, HashCode> data) {
        @Nullable
        public HashCode get(Path path) {
            return (HashCode)this.data.get((Object)path);
        }

        public int count() {
            return this.data.size();
        }

        public static ProviderCache load(Path root, Path dataProviderPath) throws IOException {
            try (BufferedReader bufferedReader = Files.newBufferedReader(dataProviderPath, StandardCharsets.UTF_8);){
                String string = bufferedReader.readLine();
                if (!string.startsWith(HashCache.HEADER_MARKER)) {
                    throw new IllegalStateException("Missing cache file header");
                }
                String[] strings = string.substring(HashCache.HEADER_MARKER.length()).split("\t", 2);
                String string2 = strings[0];
                ImmutableMap.Builder builder = ImmutableMap.builder();
                bufferedReader.lines().forEach(line -> {
                    int i = line.indexOf(32);
                    builder.put((Object)root.resolve(line.substring(i + 1)), (Object)HashCode.fromString((String)line.substring(0, i)));
                });
                ProviderCache providerCache = new ProviderCache(string2, (ImmutableMap<Path, HashCode>)builder.build());
                return providerCache;
            }
        }

        public void save(Path root, Path dataProviderPath, String description) {
            try (BufferedWriter bufferedWriter = Files.newBufferedWriter(dataProviderPath, StandardCharsets.UTF_8, new OpenOption[0]);){
                bufferedWriter.write(HashCache.HEADER_MARKER);
                bufferedWriter.write(this.version);
                bufferedWriter.write(9);
                bufferedWriter.write(description);
                bufferedWriter.newLine();
                for (Map.Entry entry : this.data.entrySet()) {
                    bufferedWriter.write(((HashCode)entry.getValue()).toString());
                    bufferedWriter.write(32);
                    bufferedWriter.write(root.relativize((Path)entry.getKey()).toString());
                    bufferedWriter.newLine();
                }
            }
            catch (IOException iOException) {
                LOGGER.warn("Unable write cachefile {}: {}", (Object)dataProviderPath, (Object)iOException);
            }
        }
    }

    class CacheUpdater
    implements CachedOutput {
        private final String provider;
        private final ProviderCache oldCache;
        private final ProviderCacheBuilder newCache;
        private final AtomicInteger writes = new AtomicInteger();
        private volatile boolean closed;

        CacheUpdater(HashCache hashCache, String providerName, String version, ProviderCache oldCache) {
            this.provider = providerName;
            this.oldCache = oldCache;
            this.newCache = new ProviderCacheBuilder(version);
        }

        private boolean shouldWrite(Path path, HashCode hashCode) {
            return !Objects.equals(this.oldCache.get(path), hashCode) || !Files.exists(path, new LinkOption[0]);
        }

        @Override
        @Override
        public void writeIfNeeded(Path path, byte[] data, HashCode hashCode) throws IOException {
            if (this.closed) {
                throw new IllegalStateException("Cannot write to cache as it has already been closed");
            }
            if (this.shouldWrite(path, hashCode)) {
                this.writes.incrementAndGet();
                Files.createDirectories(path.getParent(), new FileAttribute[0]);
                Files.write(path, data, new OpenOption[0]);
            }
            this.newCache.put(path, hashCode);
        }

        public UpdateResult close() {
            this.closed = true;
            return new UpdateResult(this.provider, this.newCache.build(), this.writes.get());
        }
    }

    @FunctionalInterface
    public static interface UpdateFunction {
        public CompletableFuture<?> update(CachedOutput var1);
    }

    public record UpdateResult(String providerId, ProviderCache cache, int writes) {
    }

    record ProviderCacheBuilder(String version, ConcurrentMap<Path, HashCode> data) {
        ProviderCacheBuilder(String version) {
            this(version, new ConcurrentHashMap<Path, HashCode>());
        }

        public void put(Path path, HashCode hashCode) {
            this.data.put(path, hashCode);
        }

        public ProviderCache build() {
            return new ProviderCache(this.version, (ImmutableMap<Path, HashCode>)ImmutableMap.copyOf(this.data));
        }
    }
}

