/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.world.level.chunk.storage;

import com.destroystokyo.paper.exception.ServerInternalException;
import com.destroystokyo.paper.util.SneakyThrow;
import com.google.common.annotations.VisibleForTesting;
import com.mojang.logging.LogUtils;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Random;
import java.util.concurrent.locks.ReentrantLock;
import java.util.zip.InflaterInputStream;
import javax.annotation.Nullable;
import net.minecraft.Util;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtIo;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.profiling.jfr.JvmProfiler;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.chunk.status.ChunkStatus;
import net.minecraft.world.level.chunk.storage.ChunkSerializer;
import net.minecraft.world.level.chunk.storage.RegionBitmap;
import net.minecraft.world.level.chunk.storage.RegionFileStorage;
import net.minecraft.world.level.chunk.storage.RegionFileVersion;
import net.minecraft.world.level.chunk.storage.RegionStorageInfo;
import org.leavesmc.leaves.region.AbstractRegionFile;
import org.slf4j.Logger;

public class RegionFile
implements AutoCloseable,
AbstractRegionFile {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final int SECTOR_BYTES = 4096;
    @VisibleForTesting
    protected static final int SECTOR_INTS = 1024;
    private static final int CHUNK_HEADER_SIZE = 5;
    private static final int HEADER_OFFSET = 0;
    private static final ByteBuffer PADDING_BUFFER = ByteBuffer.allocateDirect(1);
    private static final String EXTERNAL_FILE_EXTENSION = ".mcc";
    private static final int EXTERNAL_STREAM_FLAG = 128;
    private static final int EXTERNAL_CHUNK_THRESHOLD = 256;
    private static final int CHUNK_NOT_PRESENT = 0;
    final RegionStorageInfo info;
    private final Path path;
    private final FileChannel file;
    private final Path externalFileDir;
    final RegionFileVersion version;
    private final ByteBuffer header;
    private final IntBuffer offsets;
    private final IntBuffer timestamps;
    @VisibleForTesting
    protected final RegionBitmap usedSectors;
    public final ReentrantLock fileLock = new ReentrantLock();
    private static final CompoundTag OVERSIZED_COMPOUND = new CompoundTag();
    final boolean canRecalcHeader;
    private final ChunkStatus[] statuses = new ChunkStatus[1024];
    private boolean closed;
    public static final int MAX_CHUNK_SIZE = 524288000;
    private final byte[] oversized = new byte[1024];
    private int oversizedCount;

    private static long roundToSectors(long bytes) {
        long sectors = bytes >>> 12;
        long remainingBytes = bytes & 0xFFFL;
        long sign = -remainingBytes;
        return sectors + (sign >>> 63);
    }

    @Override
    public Path getRegionFile() {
        return this.path;
    }

    @Override
    public ReentrantLock getFileLock() {
        return this.fileLock;
    }

    private CompoundTag attemptRead(long sector, int chunkDataLength, long fileLength) throws IOException {
        try {
            if (chunkDataLength < 0) {
                return null;
            }
            long offset = sector * 4096L + 4L;
            if (offset + (long)chunkDataLength > fileLength) {
                return null;
            }
            ByteBuffer chunkData = ByteBuffer.allocate(chunkDataLength);
            if (chunkDataLength != this.file.read(chunkData, offset)) {
                return null;
            }
            ((Buffer)chunkData).flip();
            byte compressionType = chunkData.get();
            if (compressionType < 0) {
                return OVERSIZED_COMPOUND;
            }
            RegionFileVersion compression = RegionFileVersion.fromId(compressionType);
            if (compression == null) {
                return null;
            }
            InputStream input = compression.wrap(new ByteArrayInputStream(chunkData.array(), chunkData.position(), chunkDataLength - chunkData.position()));
            return NbtIo.read(new DataInputStream(input));
        }
        catch (Exception ex) {
            return null;
        }
    }

    private int getLength(long sector) throws IOException {
        ByteBuffer length = ByteBuffer.allocate(4);
        if (4 != this.file.read(length, sector * 4096L)) {
            return -1;
        }
        return length.getInt(0);
    }

    private void backupRegionFile() {
        Path backup = this.path.getParent().resolve(String.valueOf(this.path.getFileName()) + "." + new Random().nextLong() + ".backup");
        this.backupRegionFile(backup);
    }

    private void backupRegionFile(Path to) {
        try {
            this.file.force(true);
            LOGGER.warn("Backing up regionfile \"" + String.valueOf(this.path.toAbsolutePath()) + "\" to " + String.valueOf(to.toAbsolutePath()));
            Files.copy(this.path, to, StandardCopyOption.COPY_ATTRIBUTES);
            LOGGER.warn("Backed up the regionfile to " + String.valueOf(to.toAbsolutePath()));
        }
        catch (IOException ex) {
            LOGGER.error("Failed to backup to " + String.valueOf(to.toAbsolutePath()), (Throwable)ex);
        }
    }

    private static boolean inSameRegionfile(ChunkPos first, ChunkPos second) {
        return (first.x & 0xFFFFFFE0) == (second.x & 0xFFFFFFE0) && (first.z & 0xFFFFFFE0) == (second.z & 0xFFFFFFE0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean recalculateHeader() throws IOException {
        if (!this.canRecalcHeader) {
            return false;
        }
        ChunkPos ourLowerLeftPosition = RegionFileStorage.getRegionFileCoordinates(this.path);
        if (ourLowerLeftPosition == null) {
            LOGGER.error("Unable to get chunk location of regionfile " + String.valueOf(this.path.toAbsolutePath()) + ", cannot recover header");
            return false;
        }
        RegionFile regionFile = this;
        synchronized (regionFile) {
            int chunkZ;
            int chunkX;
            LOGGER.warn("Corrupt regionfile header detected! Attempting to re-calculate header offsets for regionfile " + String.valueOf(this.path.toAbsolutePath()), new Throwable());
            this.backupRegionFile();
            CompoundTag[] compounds = new CompoundTag[1024];
            int[] rawLengths = new int[1024];
            int[] sectorOffsets = new int[1024];
            boolean[] hasAikarOversized = new boolean[1024];
            long fileLength = this.file.size();
            long totalSectors = RegionFile.roundToSectors(fileLength);
            long maxSector = Math.min(0x7FFFFFL, totalSectors);
            for (long i = 2L; i < maxSector; ++i) {
                int chunkDataLength = this.getLength(i);
                CompoundTag compound = this.attemptRead(i, chunkDataLength, fileLength);
                if (compound == null || compound == OVERSIZED_COMPOUND) continue;
                ChunkPos chunkPos = ChunkSerializer.getChunkCoordinate(compound);
                if (!RegionFile.inSameRegionfile(ourLowerLeftPosition, chunkPos)) {
                    LOGGER.error("Ignoring absolute chunk " + String.valueOf(chunkPos) + " in regionfile as it is not contained in the bounds of the regionfile '" + String.valueOf(this.path.toAbsolutePath()) + "'. It should be in regionfile (" + (chunkPos.x >> 5) + "," + (chunkPos.z >> 5) + ")");
                    continue;
                }
                int location = chunkPos.x & 0x1F | (chunkPos.z & 0x1F) << 5;
                CompoundTag compoundTag = compounds[location];
                if (compoundTag != null && ChunkSerializer.getLastWorldSaveTime(compoundTag) > ChunkSerializer.getLastWorldSaveTime(compound)) continue;
                Path aikarOversizedFile = this.getOversizedFile(chunkPos.x, chunkPos.z);
                int isAikarOversized = 0;
                if (Files.exists(aikarOversizedFile, new LinkOption[0])) {
                    try {
                        CompoundTag aikarOversizedCompound = this.getOversizedData(chunkPos.x, chunkPos.z);
                        if (ChunkSerializer.getLastWorldSaveTime(compound) == ChunkSerializer.getLastWorldSaveTime(aikarOversizedCompound)) {
                            isAikarOversized = 1;
                        }
                    }
                    catch (Exception ex) {
                        LOGGER.error("Failed to read aikar oversized data for absolute chunk (" + chunkPos.x + "," + chunkPos.z + ") in regionfile " + String.valueOf(this.path.toAbsolutePath()) + ", oversized data for this chunk will be lost", (Throwable)ex);
                    }
                }
                hasAikarOversized[location] = isAikarOversized;
                compounds[location] = compound;
                rawLengths[location] = chunkDataLength + 4;
                sectorOffsets[location] = (int)i;
                int chunkSectorLength = (int)RegionFile.roundToSectors(rawLengths[location]);
                i += (long)chunkSectorLength;
                --i;
            }
            Path containingFolder = this.externalFileDir;
            Path[] regionFiles = (Path[])Files.list(containingFolder).toArray(Path[]::new);
            boolean[] oversized = new boolean[1024];
            RegionFileVersion[] oversizedCompressionTypes = new RegionFileVersion[1024];
            if (regionFiles != null) {
                int lowerXBound = ourLowerLeftPosition.x;
                int lowerZBound = ourLowerLeftPosition.z;
                int upperXBound = lowerXBound + 32 - 1;
                int upperZBound = lowerZBound + 32 - 1;
                for (Path regionFile2 : regionFiles) {
                    byte[] chunkData;
                    ChunkPos oversizedCoords = RegionFile.getOversizedChunkPair(regionFile2);
                    if (oversizedCoords == null || oversizedCoords.x < lowerXBound || oversizedCoords.x > upperXBound || oversizedCoords.z < lowerZBound || oversizedCoords.z > upperZBound) continue;
                    int location = oversizedCoords.x & 0x1F | (oversizedCoords.z & 0x1F) << 5;
                    try {
                        chunkData = Files.readAllBytes(regionFile2);
                    }
                    catch (Exception ex) {
                        LOGGER.error("Failed to read oversized chunk data in file " + String.valueOf(regionFile2.toAbsolutePath()) + ", data will be lost", (Throwable)ex);
                        continue;
                    }
                    CompoundTag compound = null;
                    RegionFileVersion compression = null;
                    for (RegionFileVersion compressionType : RegionFileVersion.VERSIONS.values()) {
                        try {
                            DataInputStream in = new DataInputStream(compressionType.wrap(new ByteArrayInputStream(chunkData)));
                            compound = NbtIo.read(in);
                            compression = compressionType;
                            break;
                        }
                        catch (Exception ex) {
                        }
                    }
                    if (compound == null) {
                        LOGGER.error("Failed to read oversized chunk data in file " + String.valueOf(regionFile2.toAbsolutePath()) + ", it's corrupt. Its data will be lost");
                        continue;
                    }
                    if (!ChunkSerializer.getChunkCoordinate(compound).equals(oversizedCoords)) {
                        LOGGER.error("Can't use oversized chunk stored in " + String.valueOf(regionFile2.toAbsolutePath()) + ", got absolute chunkpos: " + String.valueOf(ChunkSerializer.getChunkCoordinate(compound)) + ", expected " + String.valueOf(oversizedCoords));
                        continue;
                    }
                    if (compounds[location] != null && ChunkSerializer.getLastWorldSaveTime(compound) <= ChunkSerializer.getLastWorldSaveTime(compounds[location])) continue;
                    oversized[location] = true;
                    oversizedCompressionTypes[location] = compression;
                }
            }
            int[] calculatedOffsets = new int[1024];
            RegionBitmap newSectorAllocations = new RegionBitmap();
            newSectorAllocations.force(0, 2);
            for (chunkX = 0; chunkX < 32; ++chunkX) {
                for (chunkZ = 0; chunkZ < 32; ++chunkZ) {
                    int n = chunkX | chunkZ << 5;
                    if (oversized[n]) continue;
                    int sectorOffset = sectorOffsets[n];
                    int rawLength = rawLengths[n];
                    int sectorLength = (int)RegionFile.roundToSectors(rawLength);
                    if (newSectorAllocations.tryAllocate(sectorOffset, sectorLength)) {
                        calculatedOffsets[n] = sectorOffset << 8 | (sectorLength > 255 ? 255 : sectorLength);
                        continue;
                    }
                    LOGGER.error("Failed to allocate space for local chunk (overlapping data??) at (" + chunkX + "," + chunkZ + ") in regionfile " + String.valueOf(this.path.toAbsolutePath()) + ", chunk will be regenerated");
                }
            }
            for (chunkX = 0; chunkX < 32; ++chunkX) {
                for (chunkZ = 0; chunkZ < 32; ++chunkZ) {
                    int n = chunkX | chunkZ << 5;
                    if (!oversized[n]) continue;
                    int sectorOffset = newSectorAllocations.allocate(1);
                    int sectorLength = 1;
                    try {
                        this.file.write(this.createExternalStub(oversizedCompressionTypes[n]), sectorOffset * 4096);
                        calculatedOffsets[n] = sectorOffset << 8 | (sectorLength > 255 ? 255 : sectorLength);
                        continue;
                    }
                    catch (IOException ex) {
                        newSectorAllocations.free(sectorOffset, sectorLength);
                        LOGGER.error("Failed to write new oversized chunk data holder, local chunk at (" + chunkX + "," + chunkZ + ") in regionfile " + String.valueOf(this.path.toAbsolutePath()) + " will be regenerated");
                    }
                }
            }
            this.oversizedCount = 0;
            for (chunkX = 0; chunkX < 32; ++chunkX) {
                for (chunkZ = 0; chunkZ < 32; ++chunkZ) {
                    int n = chunkX | chunkZ << 5;
                    int isAikarOversized = hasAikarOversized[n] ? 1 : 0;
                    this.oversizedCount += isAikarOversized;
                    this.oversized[n] = (byte)isAikarOversized;
                }
            }
            if (this.oversizedCount > 0) {
                try {
                    this.writeOversizedMeta();
                }
                catch (Exception ex) {
                    LOGGER.error("Failed to write aikar oversized chunk meta, all aikar style oversized chunk data will be lost for regionfile " + String.valueOf(this.path.toAbsolutePath()), (Throwable)ex);
                    Files.deleteIfExists(this.getOversizedMetaFile());
                }
            } else {
                Files.deleteIfExists(this.getOversizedMetaFile());
            }
            this.usedSectors.copyFrom(newSectorAllocations);
            LOGGER.info("Starting summary of changes for regionfile " + String.valueOf(this.path.toAbsolutePath()));
            for (int chunkX2 = 0; chunkX2 < 32; ++chunkX2) {
                for (chunkZ = 0; chunkZ < 32; ++chunkZ) {
                    int newOffset;
                    int n = chunkX2 | chunkZ << 5;
                    int oldOffset = this.offsets.get(n);
                    if (oldOffset == (newOffset = calculatedOffsets[n])) continue;
                    this.offsets.put(n, newOffset);
                    if (oldOffset == 0) {
                        LOGGER.info("Found missing data for local chunk (" + chunkX2 + "," + chunkZ + ") in regionfile " + String.valueOf(this.path.toAbsolutePath()));
                        continue;
                    }
                    if (newOffset == 0) {
                        LOGGER.warn("Data for local chunk (" + chunkX2 + "," + chunkZ + ") could not be recovered in regionfile " + String.valueOf(this.path.toAbsolutePath()) + ", it will be regenerated");
                        continue;
                    }
                    LOGGER.info("Local chunk (" + chunkX2 + "," + chunkZ + ") changed to point to newer data or correct chunk in regionfile " + String.valueOf(this.path.toAbsolutePath()));
                }
            }
            LOGGER.info("End of change summary for regionfile " + String.valueOf(this.path.toAbsolutePath()));
            for (int i = 0; i < 1024; ++i) {
                this.timestamps.put(i, calculatedOffsets[i] != 0 ? (int)System.currentTimeMillis() : 0);
            }
            try {
                this.flush();
                this.file.force(true);
                LOGGER.info("Successfully wrote new header to disk for regionfile " + String.valueOf(this.path.toAbsolutePath()));
            }
            catch (IOException ex) {
                LOGGER.error("Failed to write new header to disk for regionfile " + String.valueOf(this.path.toAbsolutePath()), (Throwable)ex);
            }
        }
        return true;
    }

    @Override
    public void setStatus(int x, int z, ChunkStatus status) {
        if (this.closed) {
            throw new IllegalStateException("RegionFile is closed");
        }
        this.statuses[RegionFile.getChunkLocation((int)x, (int)z)] = status;
    }

    @Override
    public ChunkStatus getStatusIfCached(int x, int z) {
        if (this.closed) {
            throw new IllegalStateException("RegionFile is closed");
        }
        int location = RegionFile.getChunkLocation(x, z);
        return this.statuses[location];
    }

    public RegionFile(RegionStorageInfo storageKey, Path directory, Path path, boolean dsync) throws IOException {
        this(storageKey, directory, path, RegionFileVersion.getCompressionFormat(), dsync);
    }

    public RegionFile(RegionStorageInfo storageKey, Path directory, Path path, boolean dsync, boolean canRecalcHeader) throws IOException {
        this(storageKey, directory, path, RegionFileVersion.getCompressionFormat(), dsync, canRecalcHeader);
    }

    public RegionFile(RegionStorageInfo storageKey, Path path, Path directory, RegionFileVersion compressionFormat, boolean dsync) throws IOException {
        this(storageKey, path, directory, compressionFormat, dsync, true);
    }

    public RegionFile(RegionStorageInfo storageKey, Path path, Path directory, RegionFileVersion compressionFormat, boolean dsync, boolean canRecalcHeader) throws IOException {
        this.canRecalcHeader = canRecalcHeader;
        this.header = ByteBuffer.allocateDirect(8192);
        this.usedSectors = new RegionBitmap();
        this.info = storageKey;
        this.path = path;
        this.initOversizedState();
        this.version = compressionFormat;
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            throw new IllegalArgumentException("Expected directory, got " + String.valueOf(directory.toAbsolutePath()));
        }
        this.externalFileDir = directory;
        this.offsets = this.header.asIntBuffer();
        ((Buffer)this.offsets).limit(1024);
        ((Buffer)this.header).position(4096);
        this.timestamps = this.header.asIntBuffer();
        this.file = dsync ? FileChannel.open(path, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.DSYNC) : FileChannel.open(path, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        this.usedSectors.force(0, 2);
        ((Buffer)this.header).position(0);
        int i = this.file.read(this.header, 0L);
        if (i != -1) {
            long j;
            if (i != 8192) {
                LOGGER.warn("Region file {} has truncated header: {}", (Object)path, (Object)i);
            }
            long regionFileSize = j = Files.size(path);
            boolean needsHeaderRecalc = false;
            boolean hasBackedUp = false;
            for (int k = 0; k < 1024; ++k) {
                boolean failedToAllocate;
                int i1;
                int headerLocation = k;
                int l = this.offsets.get(k);
                if (l == 0) continue;
                int offset = i1 = RegionFile.getSectorNumber(l);
                int j1 = RegionFile.getNumSectors(l);
                if (j1 == 255) {
                    ByteBuffer realLen = ByteBuffer.allocate(4);
                    this.file.read(realLen, i1 * 4096);
                    j1 = (realLen.getInt(0) + 4) / 4096 + 1;
                }
                int sectorLength = j1;
                if (i1 < 2) {
                    LOGGER.warn("Region file {} has invalid sector at index: {}; sector {} overlaps with header", new Object[]{path, k, i1});
                } else if (j1 == 0) {
                    LOGGER.warn("Region file {} has an invalid sector at index: {}; size has to be > 0", (Object)path, (Object)k);
                } else if ((long)i1 * 4096L > j) {
                    LOGGER.warn("Region file {} has an invalid sector at index: {}; sector {} is out of bounds", new Object[]{path, k, i1});
                }
                if (offset < 2 || sectorLength <= 0 || (long)offset * 4096L > regionFileSize) {
                    if (canRecalcHeader) {
                        LOGGER.error("Detected invalid header for regionfile " + String.valueOf(this.path.toAbsolutePath()) + "! Recalculating header...");
                        needsHeaderRecalc = true;
                        break;
                    }
                    LOGGER.error("Detected invalid header for regionfile " + String.valueOf(this.path.toAbsolutePath()) + "! Cannot recalculate, removing local chunk (" + (headerLocation & 0x1F) + "," + (headerLocation >>> 5) + ") from header");
                    if (!hasBackedUp) {
                        hasBackedUp = true;
                        this.backupRegionFile();
                    }
                    this.timestamps.put(headerLocation, 0);
                    this.offsets.put(headerLocation, 0);
                    continue;
                }
                boolean bl = failedToAllocate = !this.usedSectors.tryAllocate(offset, sectorLength);
                if (failedToAllocate) {
                    LOGGER.error("Overlapping allocation by local chunk (" + (headerLocation & 0x1F) + "," + (headerLocation >>> 5) + ") in regionfile " + String.valueOf(this.path.toAbsolutePath()));
                }
                if (failedToAllocate & !canRecalcHeader) {
                    LOGGER.error("Detected invalid header for regionfile " + String.valueOf(this.path.toAbsolutePath()) + "! Cannot recalculate, removing local chunk (" + (headerLocation & 0x1F) + "," + (headerLocation >>> 5) + ") from header");
                    if (!hasBackedUp) {
                        hasBackedUp = true;
                        this.backupRegionFile();
                    }
                    this.timestamps.put(headerLocation, 0);
                    this.offsets.put(headerLocation, 0);
                    continue;
                }
                needsHeaderRecalc |= failedToAllocate;
            }
            if (needsHeaderRecalc) {
                LOGGER.error("Recalculating regionfile " + String.valueOf(this.path.toAbsolutePath()) + ", header gave erroneous offsets & locations");
                this.recalculateHeader();
            }
        }
    }

    public Path getPath() {
        return this.path;
    }

    private Path getExternalChunkPath(ChunkPos chunkPos) {
        String s = "c." + chunkPos.x + "." + chunkPos.z + EXTERNAL_FILE_EXTENSION;
        return this.externalFileDir.resolve(s);
    }

    private static ChunkPos getOversizedChunkPair(Path file) {
        String fileName = file.getFileName().toString();
        if (!fileName.startsWith("c.") || !fileName.endsWith(EXTERNAL_FILE_EXTENSION)) {
            return null;
        }
        String[] split = fileName.split("\\.");
        if (split.length != 4) {
            return null;
        }
        try {
            int x = Integer.parseInt(split[1]);
            int z = Integer.parseInt(split[2]);
            return new ChunkPos(x, z);
        }
        catch (NumberFormatException ex) {
            return null;
        }
    }

    @Override
    @Nullable
    public synchronized DataInputStream getChunkDataInputStream(ChunkPos pos) throws IOException {
        int i = this.getOffset(pos);
        if (i == 0) {
            return null;
        }
        int j = RegionFile.getSectorNumber(i);
        int k = RegionFile.getNumSectors(i);
        if (k == 255) {
            ByteBuffer realLen = ByteBuffer.allocate(4);
            this.file.read(realLen, j * 4096);
            k = (realLen.getInt(0) + 4) / 4096 + 1;
        }
        int l = k * 4096;
        ByteBuffer bytebuffer = ByteBuffer.allocate(l);
        this.file.read(bytebuffer, j * 4096);
        ((Buffer)bytebuffer).flip();
        if (bytebuffer.remaining() < 5) {
            LOGGER.error("Chunk {} header is truncated: expected {} but read {}", new Object[]{pos, l, bytebuffer.remaining()});
            if (this.canRecalcHeader && this.recalculateHeader()) {
                return this.getChunkDataInputStream(pos);
            }
            return null;
        }
        int i1 = bytebuffer.getInt();
        byte b0 = bytebuffer.get();
        if (i1 == 0) {
            LOGGER.warn("Chunk {} is allocated, but stream is missing", (Object)pos);
            if (this.canRecalcHeader && this.recalculateHeader()) {
                return this.getChunkDataInputStream(pos);
            }
            return null;
        }
        int j1 = i1 - 1;
        if (RegionFile.isExternalStreamChunk(b0)) {
            DataInputStream ret;
            if (j1 != 0) {
                LOGGER.warn("Chunk has both internal and external streams");
                if (this.canRecalcHeader && this.recalculateHeader()) {
                    return this.getChunkDataInputStream(pos);
                }
            }
            if ((ret = this.createExternalChunkInputStream(pos, RegionFile.getExternalChunkVersion(b0))) == null && this.canRecalcHeader && this.recalculateHeader()) {
                return this.getChunkDataInputStream(pos);
            }
            return ret;
        }
        if (j1 > bytebuffer.remaining()) {
            LOGGER.error("Chunk {} stream is truncated: expected {} but read {}", new Object[]{pos, j1, bytebuffer.remaining()});
            if (this.canRecalcHeader && this.recalculateHeader()) {
                return this.getChunkDataInputStream(pos);
            }
            return null;
        }
        if (j1 < 0) {
            LOGGER.error("Declared size {} of chunk {} is negative", (Object)i1, (Object)pos);
            if (this.canRecalcHeader && this.recalculateHeader()) {
                return this.getChunkDataInputStream(pos);
            }
            return null;
        }
        JvmProfiler.INSTANCE.onRegionFileRead(this.info, pos, this.version, j1);
        DataInputStream ret = this.createChunkInputStream(pos, b0, RegionFile.createStream(bytebuffer, j1));
        if (ret == null && this.canRecalcHeader && this.recalculateHeader()) {
            return this.getChunkDataInputStream(pos);
        }
        return ret;
    }

    private static int getTimestamp() {
        return (int)(Util.getEpochMillis() / 1000L);
    }

    private static boolean isExternalStreamChunk(byte flags) {
        return (flags & 0x80) != 0;
    }

    private static byte getExternalChunkVersion(byte flags) {
        return (byte)(flags & 0xFFFFFF7F);
    }

    @Nullable
    private DataInputStream createChunkInputStream(ChunkPos pos, byte flags, InputStream stream) throws IOException {
        RegionFileVersion regionfilecompression = RegionFileVersion.fromId(flags);
        if (regionfilecompression == RegionFileVersion.VERSION_CUSTOM) {
            String s = new DataInputStream(stream).readUTF();
            ResourceLocation minecraftkey = ResourceLocation.tryParse(s);
            if (minecraftkey != null) {
                LOGGER.error("Unrecognized custom compression {}", (Object)minecraftkey);
                return null;
            }
            LOGGER.error("Invalid custom compression id {}", (Object)s);
            return null;
        }
        if (regionfilecompression == null) {
            LOGGER.error("Chunk {} has invalid chunk stream version {}", (Object)pos, (Object)flags);
            return null;
        }
        return new DataInputStream(regionfilecompression.wrap(stream));
    }

    @Nullable
    private DataInputStream createExternalChunkInputStream(ChunkPos pos, byte flags) throws IOException {
        Path path = this.getExternalChunkPath(pos);
        if (!Files.isRegularFile(path, new LinkOption[0])) {
            LOGGER.error("External chunk path {} is not file", (Object)path);
            return null;
        }
        return this.createChunkInputStream(pos, flags, Files.newInputStream(path, new OpenOption[0]));
    }

    private static ByteArrayInputStream createStream(ByteBuffer buffer, int length) {
        return new ByteArrayInputStream(buffer.array(), buffer.position(), length);
    }

    private int packSectorOffset(int offset, int size) {
        return offset << 8 | size;
    }

    private static int getNumSectors(int sectorData) {
        return sectorData & 0xFF;
    }

    private static int getSectorNumber(int sectorData) {
        return sectorData >> 8 & 0xFFFFFF;
    }

    private static int sizeToSectors(int byteCount) {
        return (byteCount + 4096 - 1) / 4096;
    }

    @Override
    public synchronized boolean doesChunkExist(ChunkPos pos) {
        int i = this.getOffset(pos);
        if (i == 0) {
            return false;
        }
        int j = RegionFile.getSectorNumber(i);
        int k = RegionFile.getNumSectors(i);
        ByteBuffer bytebuffer = ByteBuffer.allocate(5);
        try {
            this.file.read(bytebuffer, j * 4096);
            ((Buffer)bytebuffer).flip();
            if (bytebuffer.remaining() != 5) {
                return false;
            }
            int l = bytebuffer.getInt();
            byte b0 = bytebuffer.get();
            if (RegionFile.isExternalStreamChunk(b0)) {
                if (!RegionFileVersion.isValidVersion(RegionFile.getExternalChunkVersion(b0))) {
                    return false;
                }
                if (!Files.isRegularFile(this.getExternalChunkPath(pos), new LinkOption[0])) {
                    return false;
                }
            } else {
                if (!RegionFileVersion.isValidVersion(b0)) {
                    return false;
                }
                if (l == 0) {
                    return false;
                }
                int i1 = l - 1;
                if (i1 < 0 || i1 > 4096 * k) {
                    return false;
                }
            }
            return true;
        }
        catch (IOException ioexception) {
            SneakyThrow.sneaky((Throwable)ioexception);
            return false;
        }
    }

    @Override
    public DataOutputStream getChunkDataOutputStream(ChunkPos pos) throws IOException {
        return new DataOutputStream(this.version.wrap(new ChunkBuffer(pos)));
    }

    @Override
    public void flush() throws IOException {
        this.file.force(true);
    }

    @Override
    public void clear(ChunkPos pos) throws IOException {
        int i = RegionFile.getOffsetIndex(pos);
        int j = this.offsets.get(i);
        if (j != 0) {
            this.offsets.put(i, 0);
            this.timestamps.put(i, RegionFile.getTimestamp());
            this.writeHeader();
            Files.deleteIfExists(this.getExternalChunkPath(pos));
            this.usedSectors.free(RegionFile.getSectorNumber(j), RegionFile.getNumSectors(j));
        }
    }

    protected synchronized void write(ChunkPos pos, ByteBuffer buf) throws IOException {
        CommitOp regionfile_b;
        int k1;
        int i = RegionFile.getOffsetIndex(pos);
        int j = this.offsets.get(i);
        int k = RegionFile.getSectorNumber(j);
        int l = RegionFile.getNumSectors(j);
        int i1 = buf.remaining();
        int j1 = RegionFile.sizeToSectors(i1);
        if (j1 >= 256) {
            Path path = this.getExternalChunkPath(pos);
            LOGGER.warn("Saving oversized chunk {} ({} bytes} to external file {}", new Object[]{pos, i1, path});
            j1 = 1;
            k1 = this.usedSectors.allocate(j1);
            regionfile_b = this.writeToExternalFile(path, buf);
            ByteBuffer bytebuffer1 = this.createExternalStub();
            this.file.write(bytebuffer1, k1 * 4096);
        } else {
            k1 = this.usedSectors.allocate(j1);
            regionfile_b = () -> Files.deleteIfExists(this.getExternalChunkPath(pos));
            this.file.write(buf, k1 * 4096);
        }
        this.offsets.put(i, this.packSectorOffset(k1, j1));
        this.timestamps.put(i, RegionFile.getTimestamp());
        this.writeHeader();
        regionfile_b.run();
        if (k != 0) {
            this.usedSectors.free(k, l);
        }
    }

    private ByteBuffer createExternalStub() {
        return this.createExternalStub(this.version);
    }

    private ByteBuffer createExternalStub(RegionFileVersion version) {
        ByteBuffer bytebuffer = ByteBuffer.allocate(5);
        bytebuffer.putInt(1);
        bytebuffer.put((byte)(version.getId() | 0x80));
        ((Buffer)bytebuffer).flip();
        return bytebuffer;
    }

    private CommitOp writeToExternalFile(Path path, ByteBuffer buf) throws IOException {
        Path path1 = Files.createTempFile(this.externalFileDir, "tmp", (String)null, new FileAttribute[0]);
        FileChannel filechannel = FileChannel.open(path1, StandardOpenOption.CREATE, StandardOpenOption.WRITE);
        try {
            ((Buffer)buf).position(5);
            filechannel.write(buf);
        }
        catch (Throwable throwable) {
            ServerInternalException.reportInternalException((Throwable)throwable);
            if (filechannel != null) {
                try {
                    filechannel.close();
                }
                catch (Throwable throwable1) {
                    throwable.addSuppressed(throwable1);
                }
            }
            throw throwable;
        }
        if (filechannel != null) {
            filechannel.close();
        }
        return () -> Files.move(path1, path, StandardCopyOption.REPLACE_EXISTING);
    }

    private void writeHeader() throws IOException {
        ((Buffer)this.header).position(0);
        this.file.write(this.header, 0L);
    }

    private int getOffset(ChunkPos pos) {
        return this.offsets.get(RegionFile.getOffsetIndex(pos));
    }

    @Override
    public boolean hasChunk(ChunkPos pos) {
        return this.getOffset(pos) != 0;
    }

    private static int getChunkLocation(int x, int z) {
        return (x & 0x1F) + (z & 0x1F) * 32;
    }

    private static int getOffsetIndex(ChunkPos pos) {
        return pos.getRegionLocalX() + pos.getRegionLocalZ() * 32;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        this.fileLock.lock();
        RegionFile regionFile = this;
        synchronized (regionFile) {
            try {
                this.closed = true;
                try {
                    this.padToFullSector();
                }
                finally {
                    try {
                        this.file.force(true);
                    }
                    finally {
                        this.file.close();
                    }
                }
            }
            finally {
                this.fileLock.unlock();
            }
        }
    }

    private void padToFullSector() throws IOException {
        int j;
        int i = (int)this.file.size();
        if (i != (j = RegionFile.sizeToSectors(i) * 4096)) {
            ByteBuffer bytebuffer = PADDING_BUFFER.duplicate();
            ((Buffer)bytebuffer).position(0);
            this.file.write(bytebuffer, j - 1);
        }
    }

    private synchronized void initOversizedState() throws IOException {
        Path metaFile = this.getOversizedMetaFile();
        if (Files.exists(metaFile, new LinkOption[0])) {
            byte[] read = Files.readAllBytes(metaFile);
            System.arraycopy(read, 0, this.oversized, 0, this.oversized.length);
            for (byte temp : this.oversized) {
                this.oversizedCount += temp;
            }
        }
    }

    private static int getChunkIndex(int x, int z) {
        return (x & 0x1F) + (z & 0x1F) * 32;
    }

    @Override
    public synchronized boolean isOversized(int x, int z) {
        return this.oversized[RegionFile.getChunkIndex(x, z)] == 1;
    }

    @Override
    public synchronized void setOversized(int x, int z, boolean oversized) throws IOException {
        Path oversizedMetaFile;
        Path oversizedFile;
        int offset = RegionFile.getChunkIndex(x, z);
        boolean previous = this.oversized[offset] == 1;
        this.oversized[offset] = (byte)(oversized ? 1 : 0);
        if (!previous && oversized) {
            ++this.oversizedCount;
        } else if (!oversized && previous) {
            --this.oversizedCount;
        }
        if (previous && !oversized && Files.exists(oversizedFile = this.getOversizedFile(x, z), new LinkOption[0])) {
            Files.delete(oversizedFile);
        }
        if (this.oversizedCount > 0) {
            if (previous != oversized) {
                this.writeOversizedMeta();
            }
        } else if (previous && Files.exists(oversizedMetaFile = this.getOversizedMetaFile(), new LinkOption[0])) {
            Files.delete(oversizedMetaFile);
        }
    }

    private void writeOversizedMeta() throws IOException {
        Files.write(this.getOversizedMetaFile(), this.oversized, new OpenOption[0]);
    }

    private Path getOversizedMetaFile() {
        return this.path.getParent().resolve(this.path.getFileName().toString().replaceAll("\\.mca$", "") + ".oversized.nbt");
    }

    private Path getOversizedFile(int x, int z) {
        return this.path.getParent().resolve(this.path.getFileName().toString().replaceAll("\\.mca$", "") + "_oversized_" + x + "_" + z + ".nbt");
    }

    @Override
    public synchronized CompoundTag getOversizedData(int x, int z) throws IOException {
        Path file = this.getOversizedFile(x, z);
        try (DataInputStream out = new DataInputStream(new BufferedInputStream(new InflaterInputStream(Files.newInputStream(file, new OpenOption[0]))));){
            CompoundTag compoundTag = NbtIo.read(out);
            return compoundTag;
        }
    }

    private class ChunkBuffer
    extends ByteArrayOutputStream {
        private final ChunkPos pos;

        public ChunkBuffer(ChunkPos chunkcoordintpair) {
            super(8096);
            super.write(0);
            super.write(0);
            super.write(0);
            super.write(0);
            super.write(RegionFile.this.version.getId());
            this.pos = chunkcoordintpair;
        }

        @Override
        public void write(int b) {
            if (this.count > 524288000) {
                throw new RegionFileStorage.RegionFileSizeException("Region file too large: " + this.count);
            }
            super.write(b);
        }

        @Override
        public void write(byte[] b, int off, int len) {
            if (this.count + len > 524288000) {
                throw new RegionFileStorage.RegionFileSizeException("Region file too large: " + (this.count + len));
            }
            super.write(b, off, len);
        }

        @Override
        public void close() throws IOException {
            ByteBuffer bytebuffer = ByteBuffer.wrap(this.buf, 0, this.count);
            int i = this.count - 5 + 1;
            JvmProfiler.INSTANCE.onRegionFileWrite(RegionFile.this.info, this.pos, RegionFile.this.version, i);
            bytebuffer.putInt(0, i);
            RegionFile.this.write(this.pos, bytebuffer);
        }
    }

    private static interface CommitOp {
        public void run() throws IOException;
    }
}

