/*
 * Decompiled with CFR 0.152.
 */
package org.spongepowered.vanilla.installer;

import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Type;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.AccessDeniedException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;
import net.minecraftforge.fart.api.Renamer;
import net.minecraftforge.fart.api.SignatureStripperConfig;
import net.minecraftforge.fart.api.SourceFixerConfig;
import net.minecraftforge.fart.api.Transformer;
import net.minecraftforge.srgutils.IMappingFile;
import org.spongepowered.vanilla.installer.Agent;
import org.spongepowered.vanilla.installer.AsyncUtils;
import org.spongepowered.vanilla.installer.Installer;
import org.spongepowered.vanilla.installer.InstallerUtils;
import org.spongepowered.vanilla.installer.LauncherCommandLine;
import org.spongepowered.vanilla.installer.LibraryManager;
import org.spongepowered.vanilla.installer.model.GroupArtifactVersion;
import org.spongepowered.vanilla.installer.model.mojang.BundleElement;
import org.spongepowered.vanilla.installer.model.mojang.BundlerMetadata;
import org.spongepowered.vanilla.installer.model.mojang.FormatVersion;
import org.spongepowered.vanilla.installer.model.mojang.Version;
import org.spongepowered.vanilla.installer.model.mojang.VersionManifest;
import org.tinylog.Logger;

public final class InstallerMain {
    private static final String COLLECTION_BOOTSTRAP = "bootstrap";
    private static final String COLLECTION_MAIN = "main";
    private static final int MAX_TRIES = 2;
    private final Installer installer;

    public InstallerMain(String[] args) throws Exception {
        LauncherCommandLine.configure(args);
        this.installer = new Installer(LauncherCommandLine.installerDirectory);
    }

    public static void main(String[] args) throws Exception {
        new InstallerMain(args).run();
    }

    public void run() {
        try {
            this.downloadAndRun();
        }
        catch (Exception ex) {
            Logger.error((Throwable)ex, "Failed to download Sponge libraries and/or Minecraft");
            System.exit(2);
        }
        finally {
            this.installer.getLibraryManager().finishedProcessing();
        }
    }

    public void downloadAndRun() throws Exception {
        ServerAndLibraries remappedMinecraftJar = null;
        Version mcVersion = null;
        try {
            mcVersion = this.downloadMinecraftManifest();
        }
        catch (IOException ex) {
            remappedMinecraftJar = this.recoverFromMinecraftDownloadError(ex);
            this.installer.getLibraryManager().validate();
        }
        LibraryManager libraryManager = this.installer.getLibraryManager();
        try {
            if (mcVersion != null) {
                CompletableFuture<Path> mappingsFuture = this.downloadMappings(mcVersion, LauncherCommandLine.librariesDirectory);
                CompletableFuture<Path> originalMcFuture = this.downloadMinecraft(mcVersion, LauncherCommandLine.librariesDirectory);
                CompletionStage extractedFuture = originalMcFuture.thenApplyAsync(bundle -> this.extractBundle((Path)bundle, LauncherCommandLine.librariesDirectory), (Executor)libraryManager.preparationWorker());
                CompletionStage remappedMinecraftJarFuture = mappingsFuture.thenCombineAsync(extractedFuture, (mappings, minecraft) -> {
                    try {
                        return this.remapMinecraft((ServerAndLibraries)minecraft, (Path)mappings, this.installer.getLibraryManager().preparationWorker());
                    }
                    catch (IOException ex) {
                        return (ServerAndLibraries)AsyncUtils.sneakyThrow(ex);
                    }
                }, (Executor)libraryManager.preparationWorker());
                libraryManager.validate();
                remappedMinecraftJar = (ServerAndLibraries)((CompletableFuture)remappedMinecraftJarFuture).get();
            }
        }
        catch (ExecutionException ex) {
            Throwable cause = ex.getCause();
            remappedMinecraftJar = this.recoverFromMinecraftDownloadError(cause instanceof Exception ? (Exception)cause : ex);
        }
        assert (remappedMinecraftJar != null);
        libraryManager.addLibrary(COLLECTION_MAIN, new LibraryManager.Library("minecraft", remappedMinecraftJar.server()));
        for (Map.Entry<GroupArtifactVersion, Path> library : remappedMinecraftJar.libraries().entrySet()) {
            if (library.getKey().group().equals("com.mojang") || library.getKey().group().equals("net.minecraft")) {
                libraryManager.addLibrary(COLLECTION_MAIN, new LibraryManager.Library(library.getKey().toString(), library.getValue()));
                continue;
            }
            libraryManager.addLibrary(COLLECTION_BOOTSTRAP, new LibraryManager.Library(library.getKey().toString(), library.getValue()));
        }
        this.installer.getLibraryManager().finishedProcessing();
        Logger.info("Environment has been verified.");
        HashSet seenLibs = new HashSet();
        this.installer.getLibraryManager().getAll(COLLECTION_BOOTSTRAP).stream().peek(lib -> seenLibs.add(lib.getName())).map(LibraryManager.Library::getFile).forEach(path -> {
            Logger.debug("Adding jar {} to bootstrap classpath", path);
            Agent.addJarToClasspath(path);
        });
        Path[] transformableLibs = (Path[])this.installer.getLibraryManager().getAll(COLLECTION_MAIN).stream().filter(lib -> !seenLibs.contains(lib.getName())).map(LibraryManager.Library::getFile).toArray(Path[]::new);
        ArrayList<String> gameArgs = new ArrayList<String>(LauncherCommandLine.remainingArgs);
        Collections.addAll(gameArgs, this.installer.getLauncherConfig().args.split(" "));
        Agent.crackModules();
        String className = "org.spongepowered.vanilla.applaunch.Main";
        InstallerMain.invokeMain("org.spongepowered.vanilla.applaunch.Main", gameArgs.toArray(new String[0]), transformableLibs);
    }

    private <T extends Throwable> ServerAndLibraries recoverFromMinecraftDownloadError(T ex) throws T {
        Path expectedUnpacked = this.expectedMinecraftLocation(LauncherCommandLine.librariesDirectory, "1.19.4");
        Path expectedRemapped = this.expectedRemappedLocation(expectedUnpacked);
        if (Files.exists(expectedRemapped, new LinkOption[0])) {
            Logger.warn(ex, "Failed to download and remap Minecraft. An existing jar exists, so we will attempt to use that instead.");
            return this.extractBundle(this.expectedBundleLocation(expectedUnpacked), LauncherCommandLine.librariesDirectory);
        }
        throw ex;
    }

    private static void invokeMain(String className, String[] args, Path[] extraCpEntries) {
        try {
            Class.forName(className).getMethod(COLLECTION_MAIN, String[].class, Path[].class).invoke(null, args, extraCpEntries);
        }
        catch (InvocationTargetException ex) {
            Logger.error(ex.getCause(), "Failed to invoke main class {} due to an error", className);
            System.exit(1);
        }
        catch (ClassNotFoundException | IllegalAccessException | NoSuchMethodException ex) {
            Logger.error((Throwable)ex, "Failed to invoke main class {} due to an error", className);
            System.exit(1);
        }
    }

    private Version downloadMinecraftManifest() throws IOException {
        Version version;
        Logger.info("Downloading the Minecraft versions manifest...");
        VersionManifest.Version foundVersionManifest = null;
        Gson gson = new Gson();
        URLConnection conn = new URL("https://launchermeta.mojang.com/mc/game/version_manifest.json").openConnection();
        conn.setConnectTimeout(5000);
        try (JsonReader reader = new JsonReader(new InputStreamReader(conn.getInputStream()));){
            VersionManifest manifest = (VersionManifest)gson.fromJson(reader, (Type)((Object)VersionManifest.class));
            for (VersionManifest.Version version2 : manifest.versions) {
                if (!"1.19.4".equals(version2.id)) continue;
                foundVersionManifest = version2;
                break;
            }
        }
        if (foundVersionManifest == null) {
            throw new IOException(String.format("Failed to find version manifest for '%s'!", "1.19.4"));
        }
        try (JsonReader reader = new JsonReader(new InputStreamReader(foundVersionManifest.url.openStream()));){
            version = (Version)gson.fromJson(reader, (Type)((Object)Version.class));
        }
        if (version == null) {
            throw new IOException(String.format("Failed to download version information for '%s'!", "1.19.4"));
        }
        return version;
    }

    private Path expectedMinecraftLocation(Path librariesDirectory, String version) {
        return librariesDirectory.resolve("net/minecraft").resolve(version).resolve("minecraft_server.jar");
    }

    private Path expectedRemappedLocation(Path originalLocation) {
        return originalLocation.resolveSibling("minecraft_server-remapped.jar");
    }

    private Path expectedBundleLocation(Path originalLocation) {
        return originalLocation.resolveSibling("minecraft_server-bundle.jar");
    }

    private CompletableFuture<Path> downloadMinecraft(Version version, Path librariesDirectory) {
        return AsyncUtils.asyncFailableFuture(() -> {
            Path downloadTarget = this.expectedBundleLocation(this.expectedMinecraftLocation(librariesDirectory, version.id));
            if (Files.notExists(downloadTarget, new LinkOption[0])) {
                if (!this.installer.getLauncherConfig().autoDownloadLibraries) {
                    throw new IOException(String.format("The Minecraft jar is not located at '%s' and downloading it has been turned off.", downloadTarget));
                }
                InstallerUtils.downloadCheckHash(version.downloads.server.url, downloadTarget, MessageDigest.getInstance("SHA-1"), version.downloads.server.sha1);
            } else if (this.installer.getLauncherConfig().checkLibraryHashes) {
                Logger.info("Detected existing Minecraft Server jar, verifying hashes...");
                if (InstallerUtils.validateSha1(version.downloads.server.sha1, downloadTarget)) {
                    Logger.info("Minecraft Server jar verified!");
                } else {
                    Logger.error("Checksum verification failed: Expected {}. Deleting cached Minecraft Server jar...", version.downloads.server.sha1);
                    Files.delete(downloadTarget);
                    InstallerUtils.downloadCheckHash(version.downloads.server.url, downloadTarget, MessageDigest.getInstance("SHA-1"), version.downloads.server.sha1);
                }
            } else {
                Logger.info("Detected existing Minecraft jar. Skipping hash check as that is turned off...");
            }
            return downloadTarget;
        }, this.installer.getLibraryManager().preparationWorker());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private ServerAndLibraries extractBundle(Path bundleJar, Path librariesDirectory) {
        Path serverDestination = this.expectedMinecraftLocation(librariesDirectory, "1.19.4");
        try (JarFile bundle = new JarFile(bundleJar.toFile());){
            Optional<BundlerMetadata> metaOpt = BundlerMetadata.read(bundle);
            if (!metaOpt.isPresent()) {
                ServerAndLibraries serverAndLibraries = new ServerAndLibraries(bundleJar, Map.of());
                return serverAndLibraries;
            }
            BundlerMetadata md = metaOpt.get();
            if (!md.version().equals(new FormatVersion(1, 0))) {
                Logger.warn("Read bundler metadata from server jar with version {}, but we only support 1.0", md.version());
            }
            boolean serverExtractionNeeded = true;
            BundleElement server = md.server();
            if (Files.exists(serverDestination, new LinkOption[0]) && InstallerUtils.validateSha256(server.sha256(), serverDestination)) {
                serverExtractionNeeded = false;
            }
            if (serverExtractionNeeded) {
                ZipEntry serverEntry = bundle.getEntry(server.path());
                try (InputStream is = bundle.getInputStream(serverEntry);){
                    InstallerUtils.transferCheckHash(is, serverDestination, MessageDigest.getInstance("SHA-256"), server.sha256());
                }
            }
            HashMap<GroupArtifactVersion, Path> libs = new HashMap<GroupArtifactVersion, Path>();
            for (BundleElement library : md.libraries()) {
                GroupArtifactVersion gav = GroupArtifactVersion.parse(library.id());
                Path destination = gav.resolve(librariesDirectory).resolve(gav.artifact() + "-" + gav.version() + (String)(gav.classifier() == null ? "" : "-" + gav.classifier()) + ".jar");
                if (Files.exists(destination, new LinkOption[0]) && InstallerUtils.validateSha256(library.sha256(), destination)) {
                    libs.put(gav, destination);
                    continue;
                }
                ZipEntry entry = bundle.getEntry(library.path());
                InputStream is = bundle.getInputStream(entry);
                try {
                    InstallerUtils.transferCheckHash(is, destination, MessageDigest.getInstance("SHA-256"), library.sha256());
                    libs.put(gav, destination);
                }
                finally {
                    if (is == null) continue;
                    is.close();
                }
            }
            ServerAndLibraries serverAndLibraries = new ServerAndLibraries(serverDestination, libs);
            return serverAndLibraries;
        }
        catch (IOException | NoSuchAlgorithmException ex) {
            Logger.error((Throwable)ex, "Failed to extract bundle from {}", bundleJar);
            throw new RuntimeException(ex);
        }
    }

    private CompletableFuture<Path> downloadMappings(Version version, Path librariesDirectory) {
        return AsyncUtils.asyncFailableFuture(() -> {
            Logger.info("Setting up names for Minecraft {}", "1.19.4");
            Path downloadTarget = librariesDirectory.resolve("net/minecraft/mappings").resolve("1.19.4").resolve("server.txt");
            Version.Downloads.Download mappings = version.downloads.server_mappings;
            if (mappings == null) {
                throw new IOException(String.format("Mappings were not included in version manifest for %s", "1.19.4"));
            }
            boolean checkHashes = this.installer.getLauncherConfig().checkLibraryHashes;
            if (Files.exists(downloadTarget, new LinkOption[0])) {
                if (checkHashes) {
                    Logger.info("Detected existing mappings, verifying hashes...");
                    if (InstallerUtils.validateSha1(mappings.sha1, downloadTarget)) {
                        Logger.info("Mappings verified!");
                        return downloadTarget;
                    }
                    Logger.error("Checksum verification failed: Expected {}. Deleting cached server mappings file...", version.downloads.server.sha1);
                    Files.delete(downloadTarget);
                } else {
                    return downloadTarget;
                }
            }
            if (this.installer.getLauncherConfig().autoDownloadLibraries) {
                if (checkHashes) {
                    InstallerUtils.downloadCheckHash(mappings.url, downloadTarget, MessageDigest.getInstance("SHA-1"), mappings.sha1);
                } else {
                    InstallerUtils.download(mappings.url, downloadTarget, false);
                }
            } else {
                throw new IOException(String.format("Mappings were not located at '%s' and downloading them has been turned off.", downloadTarget));
            }
            return downloadTarget;
        }, this.installer.getLibraryManager().preparationWorker());
    }

    private ServerAndLibraries remapMinecraft(ServerAndLibraries minecraft, Path serverMappings, ExecutorService service) throws IOException {
        Logger.info("Checking if we need to remap Minecraft...");
        Path outputJar = this.expectedRemappedLocation(minecraft.server());
        Path tempOutput = outputJar.resolveSibling("minecraft_server_remapped.jar.tmp");
        if (Files.exists(outputJar, new LinkOption[0])) {
            Logger.info("Remapped Minecraft detected, skipping...");
            return minecraft.server(outputJar);
        }
        Logger.info("Remapping Minecraft. This may take a while...");
        IMappingFile mappings = IMappingFile.load(serverMappings.toFile()).reverse();
        Renamer.builder().input(minecraft.server().toFile()).output(tempOutput.toFile()).add(Transformer.parameterAnnotationFixerFactory()).add(ctx -> {
            final Transformer backing = Transformer.renamerFactory(mappings).create(ctx);
            return new Transformer(){

                @Override
                public Transformer.ClassEntry process(Transformer.ClassEntry entry) {
                    String name = entry.getName();
                    if (name.startsWith("it/unimi") || name.startsWith("com/google") || name.startsWith("com/mojang/datafixers") || name.startsWith("com/mojang/brigadier") || name.startsWith("org/apache")) {
                        return entry;
                    }
                    return backing.process(entry);
                }

                @Override
                public Transformer.ManifestEntry process(Transformer.ManifestEntry entry) {
                    return backing.process(entry);
                }

                @Override
                public Transformer.ResourceEntry process(Transformer.ResourceEntry entry) {
                    return backing.process(entry);
                }

                @Override
                public Collection<? extends Transformer.Entry> getExtras() {
                    return backing.getExtras();
                }
            };
        }).add(Transformer.recordFixerFactory()).add(Transformer.parameterAnnotationFixerFactory()).add(Transformer.sourceFixerFactory(SourceFixerConfig.JAVA)).add(Transformer.signatureStripperFactory(SignatureStripperConfig.ALL)).logger(Logger::debug).build().run();
        try {
            Files.move(tempOutput, outputJar, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (AccessDeniedException ex) {
            for (int tries = 0; tries < 2; ++tries) {
                try {
                    Thread.sleep(5 * tries);
                    Files.move(tempOutput, outputJar, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
                    continue;
                }
                catch (AccessDeniedException ex2) {
                    if (tries != 1) continue;
                    throw ex;
                }
                catch (InterruptedException exInterrupt) {
                    Thread.currentThread().interrupt();
                    throw ex;
                }
            }
        }
        return minecraft.server(outputJar);
    }

    record ServerAndLibraries(Path server, Map<GroupArtifactVersion, Path> libraries) {
        ServerAndLibraries {
            libraries = Map.copyOf(libraries);
        }

        public ServerAndLibraries server(Path server) {
            return new ServerAndLibraries(server, this.libraries);
        }
    }
}

